/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules;

import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationRule.Level.ERROR;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.EmptyTypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.PostValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;

import java.util.List;

public class PagedOperationsOperationMustDeclareResponseBodyRule extends PostValidationRule {

  public PagedOperationsOperationMustDeclareResponseBodyRule() {
    super("Paged Operations must declare a response body",
          "All paginated operations must declare a an output type.",
          ERROR);
  }

  @Override
  public List<ValidationResult> postValidate(ConnectorDescriptor connectorDescriptor,
                                             APIModel apiModel,
                                             ConnectorModel connectorModel) {

    List<APIOperationModel> nullOutputAPIOperations =
        connectorModel.getOperations().stream()
            .filter(x -> x.getOutputMetadata() == null || x.getOutputMetadata() instanceof EmptyTypeDefinition)
            .map(x -> getApiOperationModel(apiModel, x))
            .collect(toList());

    return nullOutputAPIOperations.stream()
        .filter(nullOutputOperation -> operationIsPaginated(connectorDescriptor, nullOutputOperation))
        .filter(nullOutputOperation -> !typeDefinedInDescriptor(connectorDescriptor, nullOutputOperation))
        .map(this::getValidationError)
        .collect(toList());
  }

  private APIOperationModel getApiOperationModel(APIModel apiModel, ConnectorOperation connectorOperation) {
    final APIOperationModel apiOperationModel = apiModel.getOperationsModel().stream()
        .filter(x -> connectorOperation.getPath().equalsIgnoreCase(x.getPath()))
        .filter(x -> connectorOperation.getHttpMethod().equalsIgnoreCase(x.getHttpMethod().name()))
        .findFirst().orElse(null);

    return requireNonNull(apiOperationModel);
  }

  private boolean typeDefinedInDescriptor(ConnectorDescriptor connectorDescriptor, APIOperationModel apiOperationModel) {
    OperationDescriptor operationDescriptor = connectorDescriptor.getEndpoints().stream()
        .filter(x -> x.getPath().equalsIgnoreCase(apiOperationModel.getPath()))
        .flatMap(x -> x.getOperations().stream())
        .filter(x -> x.getMethod().equalsIgnoreCase(apiOperationModel.getHttpMethod().name()))
        .findFirst().orElse(null);

    return operationDescriptor != null && isNotBlank(operationDescriptor.getOutputTypeSchema());
  }

  private boolean operationIsPaginated(ConnectorDescriptor connectorDescriptor, APIOperationModel apiOperationModel) {
    OperationDescriptor operationDescriptor = connectorDescriptor.getEndpoints().stream()
        .filter(x -> x.getPath().equalsIgnoreCase(apiOperationModel.getPath()))
        .flatMap(x -> x.getOperations().stream())
        .filter(x -> x.getMethod().equalsIgnoreCase(apiOperationModel.getHttpMethod().name()))
        .findFirst().orElse(null);

    return operationDescriptor != null && isNotBlank(operationDescriptor.getPagination());
  }

  private ValidationResult getValidationError(APIOperationModel apiOperationModel) {
    String location =
        "Operation with PATH: " + apiOperationModel.getPath() + " and METHOD: " + apiOperationModel.getHttpMethod().name();
    return new ValidationResult(this, location);
  }
}
