/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationRule.Level.ERROR;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TriggerDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.PreValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;

import java.util.List;

public class TriggerPathAndMethodMustExistInApiRule extends PreValidationRule {

  public TriggerPathAndMethodMustExistInApiRule() {
    super("Trigger PATH and METHOD must exist in the API specification.", EMPTY, ERROR);
  }

  @Override
  public List<ValidationResult> preValidate(ConnectorDescriptor connectorDescriptor, APIModel apiModel) {

    List<TriggerDescriptor> triggerDescriptors = connectorDescriptor.getTriggers();
    if (triggerDescriptors != null && !triggerDescriptors.isEmpty()) {
      return triggerDescriptors.stream()
          .filter(x -> !hasMatchingOperation(apiModel.getOperationsModel(), x))
          .map(this::getValidationError)
          .collect(toList());
    }

    return emptyList();
  }

  private static boolean hasMatchingOperation(List<APIOperationModel> operations, TriggerDescriptor triggerDescriptor) {
    return operations.stream()
        .anyMatch(
                  x -> x.getHttpMethod().name().equalsIgnoreCase(triggerDescriptor.getMethod().getName())
                      && x.getPath().equalsIgnoreCase(triggerDescriptor.getPath()));
  }

  private ValidationResult getValidationError(TriggerDescriptor triggerDescriptor) {
    String location =
        "Trigger with PATH: " + triggerDescriptor.getPath() + " and METHOD: " + triggerDescriptor.getMethod().getName();
    return new ValidationResult(this, location);
  }
}
