/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.getOperationDescriptor;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationRule.Level.ERROR;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.EndPointDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.PreValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationError;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APITypeModel;

import java.util.LinkedList;
import java.util.List;
import java.util.Optional;

public class DescriptorInputMediaTypeMustBePresentInApiRule extends PreValidationRule {

  //R018
  public DescriptorInputMediaTypeMustBePresentInApiRule() {
    super("The input media type defined in the descriptor is not present in the API spec",
          "If the API spec defines a set of input media types for an operation the Descriptor must use one of those to define the operation's default input media type.",
          ERROR);
  }

  @Override
  public List<ValidationError> preValidate(ConnectorDescriptor connectorDescriptor, APIModel apiModel) {

    List<ValidationError> allErrors = new LinkedList<>();

    //Check every operation descriptor and compare its input media type with the ones declared in the API spec
    for (EndPointDescriptor endpointDesc : connectorDescriptor.getEndpoints()) {
      List<ValidationError> errors = endpointDesc.getOperations().stream()
          .filter(opDesc -> isNotBlank(opDesc.getInputMediaType()))
          .filter(opDesc -> apiDefinesMediaTypes(apiModel, endpointDesc.getPath(), opDesc))
          .filter(opDesc -> !mediaTypePresentInApi(apiModel, endpointDesc.getPath(), opDesc))
          .map(opDesc -> getValidationErrorForOperationDescriptor(endpointDesc.getPath(), opDesc))
          .collect(toList());

      allErrors.addAll(errors);
    }

    //Check that if there is a global input media type declared, every operation supports or overrides it
    String globalMediaType = connectorDescriptor.getDefaultInputMediaType();
    if (isNotBlank(globalMediaType)) {
      List<ValidationError> globalErrors = apiModel.getOperationsModel().stream()
          .filter(apiOp -> !apiOp.getInputMetadataModel().isEmpty())
          .filter(apiOp -> apiOp.getInputMetadataModel().stream().anyMatch(y -> y.getMediaType() != null))
          .filter(apiOp -> !descriptorDefinesMediaType(connectorDescriptor, apiOp))
          .filter(apiOp -> !descriptorDefinesOperationMediaType(globalMediaType, apiOp))
          .map(apiOp -> getValidationErrorForGlobalMediaType(apiOp.getPath(), apiOp.getHttpMethod(), globalMediaType,
                                                             connectorDescriptor))
          .collect(toList());

      allErrors.addAll(globalErrors);
    }

    return allErrors;
  }

  protected boolean descriptorDefinesOperationMediaType(String globalMediaType, APIOperationModel apiOperationModel) {
    return apiOperationModel.getInputMetadataModel().stream()
        .anyMatch(y -> y.getMediaType().toString().equalsIgnoreCase(globalMediaType));
  }

  private boolean descriptorDefinesMediaType(ConnectorDescriptor connectorDescriptor, APIOperationModel apiOperation) {
    OperationDescriptor operationDescriptor = getOperationDescriptor(connectorDescriptor, apiOperation);

    return operationDescriptor != null && isNotBlank(operationDescriptor.getInputMediaType());
  }

  private boolean mediaTypePresentInApi(APIModel apiModel, String path, OperationDescriptor operationDesc) {
    String inputMediaType = operationDesc.getInputMediaType();
    String method = operationDesc.getMethod();

    return apiModel.getOperationsModel().stream()
        .filter(x -> x.getPath().equalsIgnoreCase(path) && x.getHttpMethod().equalsIgnoreCase(method))
        .flatMap(x -> x.getInputMetadataModel().stream())
        .map(APITypeModel::getMediaType)
        .anyMatch(x -> x.toString().equalsIgnoreCase(inputMediaType));
  }

  private boolean apiDefinesMediaTypes(APIModel apiModel, String path, OperationDescriptor operationDesc) {
    String method = operationDesc.getMethod();

    Optional<APIOperationModel> any = apiModel.getOperationsModel().stream()
        .filter(x -> x.getPath().equalsIgnoreCase(path) && x.getHttpMethod().equalsIgnoreCase(method))
        .filter(x -> !x.getInputMetadataModel().isEmpty())
        .findAny();

    return any.isPresent();
  }

  private ValidationError getValidationErrorForOperationDescriptor(String path, OperationDescriptor operationDescriptor) {
    String detail =
        "Operation with PATH: "
            + path
            + " and METHOD: "
            + operationDescriptor.getMethod()
            + " does not declare '"
            + operationDescriptor.getInputMediaType()
            + "' media type that is set in the operation descriptor";

    return new ValidationError(this, detail, operationDescriptor.getLocation());
  }

  private ValidationError getValidationErrorForGlobalMediaType(String path, String httpMethod, String globalMediaType,
                                                               ConnectorDescriptor connectorDescriptor) {
    String detail =
        "API Operation with PATH: "
            + path
            + " and METHOD: "
            + httpMethod.toUpperCase()
            + " does not declare '"
            + globalMediaType
            + "' media type that is defined as global media type in the connector descriptor";

    return new ValidationError(this, detail, connectorDescriptor.getLocation());
  }
}
