/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.HTTPMethod.fromString;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.HEADER;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.QUERY;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.URI;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.HTTPMethod;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.trigger.ParameterBinding;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.trigger.ParameterDataType;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.valueresolver.ValueResolver;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.valueresolver.ValueResolverParameter;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DataTypeDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ValueResolverBindingDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ValueResolverBindingParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ValueResolverDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ValueResolverParameterDescriptor;

import java.util.LinkedList;
import java.util.List;

public class ConnectorValueResolverBuilder {

  public List<ValueResolver> buildValueResolvers(ConnectorDescriptor connectorDescriptor) {
    if (connectorDescriptor.getValueResolvers() == null) {
      return emptyList();
    }
    return connectorDescriptor
        .getValueResolvers().stream()
        .map(this::buildValueResolver)
        .collect(toList());
  }

  private ValueResolver buildValueResolver(ValueResolverDescriptor valueResolverDescriptor) {
    return new ValueResolver(
                             valueResolverDescriptor.getName(),
                             valueResolverDescriptor.getPath(),
                             buildMethod(valueResolverDescriptor),
                             valueResolverDescriptor.getItems().getExtraction().getExpression(),
                             valueResolverDescriptor.getItems().getValue().getExpression(),
                             buildDisplayNameExpression(valueResolverDescriptor),
                             buildValuerResolverParameters(valueResolverDescriptor.getParameters()),
                             buildValueResolverBindings(valueResolverDescriptor.getBinding()));
  }

  private HTTPMethod buildMethod(ValueResolverDescriptor valueResolverDescriptor) {
    return fromString(valueResolverDescriptor.getMethod().getName());
  }

  private String buildDisplayNameExpression(ValueResolverDescriptor valueResolverDescriptor) {
    return valueResolverDescriptor.getItems().getDisplayName() != null
        ? valueResolverDescriptor.getItems().getDisplayName().getExpression() : null;
  }

  private List<ParameterBinding> buildValueResolverBindings(ValueResolverBindingDescriptor binding) {
    final List<ParameterBinding> parameterBindings = new LinkedList<>();

    if (binding != null) {
      parameterBindings.addAll(getParameterBindings(binding.getUriParameters(), URI));
      parameterBindings.addAll(getParameterBindings(binding.getQueryParameters(), QUERY));
      parameterBindings.addAll(getParameterBindings(binding.getHeaders(), HEADER));
    }

    return parameterBindings;
  }

  private List<ParameterBinding> getParameterBindings(List<ValueResolverBindingParameterDescriptor> uriParameters,
                                                      ParameterType uri) {
    return uriParameters.stream().map(x -> new ParameterBinding(uri, x.getName(), x.getValue().getExpression()))
        .collect(toList());
  }

  private List<ValueResolverParameter> buildValuerResolverParameters(List<ValueResolverParameterDescriptor> parameters) {
    return parameters.stream()
        .map(x -> new ValueResolverParameter(x.getName(), buildParameterType(x.getType())))
        .collect(toList());
  }

  private ParameterDataType buildParameterType(DataTypeDescriptor type) {
    return ParameterDataType.forName(type.getName());
  }
}
