/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.descriptor.parser;

import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.BaseUriDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.MultipleBaseUriDescriptor;

import java.util.ArrayList;
import java.util.List;

import javax.validation.constraints.NotNull;

import amf.client.model.domain.DialectDomainElement;

import static com.mulesoft.connectivity.rest.sdk.internal.descriptor.parser.DescriptorParserUtils.parseSingleStringProperty;

public class DescriptorBaseUriParser {

  private static final String REST_SDK_BASE_URI = "http://a.ml/vocabularies/rest-sdk#baseUri";
  private static final String REST_SDK_BASE_URI_TYPE = "http://a.ml/vocabularies/rest-sdk#baseUriType";
  private static final String REST_SDK_MULTIPLE_BASE_URI = "http://a.ml/vocabularies/rest-sdk#multipleBaseUri";
  private static final String REST_SDK_DEFAULT_BASE_URI = "http://a.ml/vocabularies/rest-sdk#defaultBaseUri";
  private static final String API_CONTRACT_URL = "http://a.ml/vocabularies.rest.sdk/apiContract#url";

  public BaseUriDescriptor parseBaseUri(DialectDomainElement encodes) {
    DialectDomainElement baseUriElement = encodes
        .getObjectPropertyUri(REST_SDK_BASE_URI)
        .stream()
        .findFirst()
        .orElse(null);

    if (baseUriElement == null) {
      return null;
    }

    return new BaseUriDescriptor(
                                 getUri(baseUriElement),
                                 getType(baseUriElement),
                                 getMultipleBaseUris(baseUriElement));
  }

  private String getType(@NotNull DialectDomainElement baseUriElement) {
    return parseSingleStringProperty(baseUriElement, REST_SDK_BASE_URI_TYPE);
  }

  private String getUri(@NotNull DialectDomainElement baseUriElement) {
    return parseSingleStringProperty(baseUriElement, API_CONTRACT_URL);
  }

  private List<MultipleBaseUriDescriptor> getMultipleBaseUris(@NotNull DialectDomainElement baseUriElement) {
    List<DialectDomainElement> multipleBaseUris = baseUriElement.getObjectPropertyUri(REST_SDK_MULTIPLE_BASE_URI);

    List<MultipleBaseUriDescriptor> multipleBaseUriDescriptorList = new ArrayList<>(multipleBaseUris.size());

    for (DialectDomainElement alternativeServerElement : multipleBaseUris) {
      String uriId = DescriptorParserUtils.parseCoreName(alternativeServerElement);
      String url = alternativeServerElement.getScalarByPropertyUri(API_CONTRACT_URL).get(0).toString();
      List<Object> defaults = alternativeServerElement.getScalarByPropertyUri(REST_SDK_DEFAULT_BASE_URI);
      boolean defaultUri = !defaults.isEmpty() && Boolean.parseBoolean(defaults.get(0).toString());

      MultipleBaseUriDescriptor multipleBaseUriDescriptor = new MultipleBaseUriDescriptor(uriId, url, defaultUri);
      multipleBaseUriDescriptorList.add(multipleBaseUriDescriptor);
    }

    return multipleBaseUriDescriptorList;
  }
}
