/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.descriptor.parser;

import static com.mulesoft.connectivity.rest.sdk.internal.descriptor.parser.DescriptorParserUtils.parseAlternativeBaseUri;
import static com.mulesoft.connectivity.rest.sdk.internal.descriptor.parser.DescriptorParserUtils.parseApiContractMethod;
import static com.mulesoft.connectivity.rest.sdk.internal.descriptor.parser.DescriptorParserUtils.parseInputMediaType;
import static com.mulesoft.connectivity.rest.sdk.internal.descriptor.parser.DescriptorParserUtils.parseInputTypeSchema;
import static com.mulesoft.connectivity.rest.sdk.internal.descriptor.parser.DescriptorParserUtils.parseNullableIgnore;
import static com.mulesoft.connectivity.rest.sdk.internal.descriptor.parser.DescriptorParserUtils.parseOutputMediaType;
import static com.mulesoft.connectivity.rest.sdk.internal.descriptor.parser.DescriptorParserUtils.parseOutputTypeSchema;
import static com.mulesoft.connectivity.rest.sdk.internal.descriptor.parser.DescriptorParserUtils.parseSingleBooleanOrNullProperty;
import static com.mulesoft.connectivity.rest.sdk.internal.descriptor.parser.DescriptorParserUtils.parseSingleStringProperty;

import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;

import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;

import amf.client.model.domain.DialectDomainElement;

public class DescriptorOperationParser {

  private static final String API_CONTRACT_SUPPORTED_OPERATION =
      "http://a.ml/vocabularies.rest.sdk/apiContract#supportedOperation";

  private static final String REST_SDK_OPERATION_NAME = "http://a.ml/vocabularies/rest-sdk#operationName";
  private static final String REST_SDK_OPERATION_DESCRIPTION = "http://a.ml/vocabularies/rest-sdk#operationDescription";
  private static final String REST_SDK_PAGINATION = "http://a.ml/vocabularies/rest-sdk#paginationReference";
  private static final String REST_SDK_VOID_OPERATION = "http://a.ml/vocabularies/rest-sdk#voidOperation";
  private static final String QUERY_PARAM_ARRAY_FORMAT = "http://a.ml/vocabularies/rest-sdk#queryParameterArrayFormat";

  private final DescriptorRequestParser descriptorRequestParser = new DescriptorRequestParser();
  private final DescriptorEncodesParser descriptorEncodesParser = new DescriptorEncodesParser();
  private final DescriptorSampleDataParser descriptorSampleDataParser = new DescriptorSampleDataParser();


  public List<OperationDescriptor> parseOperations(DialectDomainElement endpointElement, Path descriptorFilePath) {

    List<OperationDescriptor> operationDescriptors = new ArrayList<>();

    for (DialectDomainElement x : getSupportedOperations(endpointElement)) {
      operationDescriptors.add(parseOperation(x, descriptorFilePath));
    }

    return operationDescriptors;
  }

  private OperationDescriptor parseOperation(DialectDomainElement operationElement, Path descriptorFilePath) {
    Path descriptorDirectory = descriptorFilePath.getParent();

    OperationDescriptor operationDescriptor = new OperationDescriptor(parseApiContractMethod(operationElement),
                                                                      getName(operationElement),
                                                                      getDescription(operationElement),
                                                                      parseInputMediaType(operationElement),
                                                                      parseOutputMediaType(operationElement),
                                                                      descriptorRequestParser.parseRequest(operationElement,
                                                                                                           descriptorDirectory),
                                                                      parseNullableIgnore(operationElement),
                                                                      parseAlternativeBaseUri(operationElement),
                                                                      getPagination(operationElement),
                                                                      descriptorEncodesParser
                                                                          .parseSkipOutputTypeValidation(operationElement),
                                                                      parseVoidOperation(operationElement),
                                                                      parseInputTypeSchema(operationElement, descriptorDirectory),
                                                                      parseOutputTypeSchema(operationElement,
                                                                                            descriptorDirectory),
                                                                      getQueryParamArrayFormat(operationElement));
    operationDescriptor.setSampleDataDescriptor(descriptorSampleDataParser.parseRequest(operationElement));
    return operationDescriptor;
  }

  private List<DialectDomainElement> getSupportedOperations(DialectDomainElement endpointElement) {
    return endpointElement.getObjectPropertyUri(API_CONTRACT_SUPPORTED_OPERATION);
  }

  private Boolean parseVoidOperation(DialectDomainElement operationElement) {
    return parseSingleBooleanOrNullProperty(operationElement, REST_SDK_VOID_OPERATION);
  }

  private String getDescription(DialectDomainElement operationElement) {
    return parseSingleStringProperty(operationElement, REST_SDK_OPERATION_DESCRIPTION);
  }

  private String getName(DialectDomainElement operationElement) {
    return parseSingleStringProperty(operationElement, REST_SDK_OPERATION_NAME);
  }

  private String getPagination(DialectDomainElement operationElement) {
    return parseSingleStringProperty(operationElement, REST_SDK_PAGINATION);
  }

  private String getQueryParamArrayFormat(DialectDomainElement operationElement) {
    return parseSingleStringProperty(operationElement, QUERY_PARAM_ARRAY_FORMAT);
  }
}
