/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.descriptor.parser;

import static java.util.Collections.emptyList;

import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.InPaginationParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OutPaginationParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.PaginationDeclarationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.PaginationParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ExpressionDescriptor;

import java.util.ArrayList;
import java.util.List;

import javax.validation.constraints.NotNull;

import amf.client.model.domain.DialectDomainElement;

public class DescriptorPaginationParser {

  private static final String AML_REST_SDK_PAGINATION_DECLARATIONS =
      "http://a.ml/vocabularies/rest-sdk#paginationDeclarations";
  private static final String AML_REST_SDK_PAGINATION_DECLARATION = "http://a.ml/vocabularies.rest.sdk/core#name";
  private static final String AML_REST_SDK_PAGINATION_TYPE = "http://a.ml/vocabularies/rest-sdk#paginationType";
  private static final String AML_REST_SDK_PAGINATION_PARAMETERS =
      "http://a.ml/vocabularies/rest-sdk#paginationParameters";
  private static final String AML_REST_SDK_PAGINATION_OFFSET_PARAM_NAME =
      "http://a.ml/vocabularies/rest-sdk#offsetParamName";
  private static final String AML_REST_SDK_PAGINATION_INITIAL_OFFSET = "http://a.ml/vocabularies/rest-sdk#initialOffset";
  private static final String AML_REST_SDK_PAGINATION_NEXT_TOKEN_PARAM_NAME =
      "http://a.ml/vocabularies/rest-sdk#nextTokenParamName";
  private static final String AML_REST_SDK_PAGINATION_NEXT_TOKEN_EXPRESSION =
      "http://a.ml/vocabularies/rest-sdk#nextTokenExpression";
  private static final String AML_REST_SDK_PAGINATION_PAGE_NUMBER_PARAM_NAME =
      "http://a.ml/vocabularies/rest-sdk#pageNumberParamName";
  private static final String AML_REST_SDK_PAGINATION_INITIAL_PAGE_NUMBER =
      "http://a.ml/vocabularies/rest-sdk#initialPageNumber";
  private static final String AML_REST_SDK_PAGINATION_PAGING_RESPONSE_EXPRESSION =
      "http://a.ml/vocabularies/rest-sdk#pagingResponseExpression";

  private static final DescriptorExpressionParser expressionParser = new DescriptorExpressionParser();


  public List<PaginationDeclarationDescriptor> parsePaginationsDescription(DialectDomainElement encodes) {
    List<DialectDomainElement> paginationsDescriptor = encodes.getObjectPropertyUri(AML_REST_SDK_PAGINATION_DECLARATIONS);

    return paginationsDescriptor.isEmpty() ? emptyList() : getPagingMethods(paginationsDescriptor);
  }

  private List<PaginationDeclarationDescriptor> getPagingMethods(List<DialectDomainElement> paginationsDescriptor) {
    List<PaginationDeclarationDescriptor> paginationDeclarationDescriptors = new ArrayList<>(paginationsDescriptor.size());
    for (DialectDomainElement endpointsElement : paginationsDescriptor) {
      String pagingStrategyName = parsePagingStrategyName(endpointsElement);
      String pagingType = parsePagingType(endpointsElement);

      List<DialectDomainElement> parameters = endpointsElement.getObjectPropertyUri(AML_REST_SDK_PAGINATION_PARAMETERS);
      List<PaginationParameterDescriptor> paginationParameterDescriptors = new ArrayList<>(parameters.size());
      if (!parameters.isEmpty()) {
        DialectDomainElement pagingParameterItem = parameters.get(0);
        OutPaginationParameterDescriptor offsetParamNameDescriptor = parseOffsetParamName(pagingParameterItem);
        if (offsetParamNameDescriptor != null) {
          paginationParameterDescriptors.add(offsetParamNameDescriptor);
        }

        OutPaginationParameterDescriptor initialOffsetDescriptor = parseInitialOffset(pagingParameterItem);
        if (initialOffsetDescriptor != null) {
          paginationParameterDescriptors.add(initialOffsetDescriptor);
        }

        OutPaginationParameterDescriptor nextTokenParamName = parseNextTokenParamName(pagingParameterItem);
        if (nextTokenParamName != null) {
          paginationParameterDescriptors.add(nextTokenParamName);
        }

        InPaginationParameterDescriptor nextTokenExpression = parseNextTokenExpression(pagingParameterItem);
        if (nextTokenExpression != null) {
          paginationParameterDescriptors.add(nextTokenExpression);
        }

        OutPaginationParameterDescriptor pageNumberParamName =
            parsePageNumberParamName(pagingParameterItem);
        if (pageNumberParamName != null) {
          paginationParameterDescriptors.add(pageNumberParamName);
        }

        OutPaginationParameterDescriptor initialPageNumber =
            parseInitialPageNumber(pagingParameterItem);
        if (initialPageNumber != null) {
          paginationParameterDescriptors.add(initialPageNumber);
        }

        InPaginationParameterDescriptor pagingResponseExpression =
            parsePagingResponseExpression(pagingParameterItem);
        if (pagingResponseExpression != null) {
          paginationParameterDescriptors.add(pagingResponseExpression);
        }
      }

      paginationDeclarationDescriptors
          .add(new PaginationDeclarationDescriptor(pagingStrategyName, pagingType, paginationParameterDescriptors));
    }

    return paginationDeclarationDescriptors;
  }

  private OutPaginationParameterDescriptor parseOffsetParamName(@NotNull DialectDomainElement pagingParameterItem) {
    List<Object> offsetParamName = pagingParameterItem.getScalarByPropertyUri(AML_REST_SDK_PAGINATION_OFFSET_PARAM_NAME);

    return offsetParamName.isEmpty() ? null
        : new OutPaginationParameterDescriptor("offsetParamName", offsetParamName.get(0).toString());
  }

  private OutPaginationParameterDescriptor parseInitialOffset(@NotNull DialectDomainElement pagingParameterItem) {
    List<Object> initialOffset = pagingParameterItem.getScalarByPropertyUri(AML_REST_SDK_PAGINATION_INITIAL_OFFSET);

    return initialOffset.isEmpty() ? null
        : new OutPaginationParameterDescriptor("initialOffset", initialOffset.get(0).toString());
  }

  private OutPaginationParameterDescriptor parseNextTokenParamName(@NotNull DialectDomainElement pagingParameterItem) {
    List<Object> nextTokenParamName =
        pagingParameterItem.getScalarByPropertyUri(AML_REST_SDK_PAGINATION_NEXT_TOKEN_PARAM_NAME);

    return nextTokenParamName.isEmpty() ? null
        : new OutPaginationParameterDescriptor("nextTokenParamName", nextTokenParamName.get(0).toString());
  }

  private OutPaginationParameterDescriptor parsePageNumberParamName(@NotNull DialectDomainElement pagingParameterItem) {
    List<Object> pageNumberParamName =
        pagingParameterItem.getScalarByPropertyUri(AML_REST_SDK_PAGINATION_PAGE_NUMBER_PARAM_NAME);

    return pageNumberParamName.isEmpty() ? null
        : new OutPaginationParameterDescriptor("pageNumberParamName", pageNumberParamName.get(0).toString());
  }

  private OutPaginationParameterDescriptor parseInitialPageNumber(@NotNull DialectDomainElement pagingParameterItem) {
    List<Object> initialPageNumber = pagingParameterItem.getScalarByPropertyUri(AML_REST_SDK_PAGINATION_INITIAL_PAGE_NUMBER);

    return initialPageNumber.isEmpty() ? null
        : new OutPaginationParameterDescriptor("initialPageNumber", initialPageNumber.get(0).toString());
  }

  private InPaginationParameterDescriptor parseNextTokenExpression(@NotNull DialectDomainElement pagingParameterItem) {
    List<DialectDomainElement> nextTokenExpression =
        pagingParameterItem.getObjectPropertyUri(AML_REST_SDK_PAGINATION_NEXT_TOKEN_EXPRESSION);
    ExpressionDescriptor expressionDescriptor =
        parseResponseValueExtractionDescriptor(nextTokenExpression);

    return expressionDescriptor == null ? null
        : new InPaginationParameterDescriptor("nextTokenExpression", expressionDescriptor);
  }

  private InPaginationParameterDescriptor parsePagingResponseExpression(@NotNull DialectDomainElement pagingParameterItem) {
    List<DialectDomainElement> pagingResponseExpression =
        pagingParameterItem.getObjectPropertyUri(AML_REST_SDK_PAGINATION_PAGING_RESPONSE_EXPRESSION);
    ExpressionDescriptor expressionDescriptor =
        parseResponseValueExtractionDescriptor(pagingResponseExpression);

    return expressionDescriptor == null ? null
        : new InPaginationParameterDescriptor("pagingResponseExpression", expressionDescriptor);
  }

  private String parsePagingStrategyName(@NotNull DialectDomainElement endpointsElement) {
    List<Object> pagingStrategyNames = endpointsElement.getScalarByPropertyUri(AML_REST_SDK_PAGINATION_DECLARATION);

    return pagingStrategyNames.isEmpty() ? null : pagingStrategyNames.get(0).toString();
  }

  private String parsePagingType(@NotNull DialectDomainElement endpointsElement) {
    List<Object> pagingTypes = endpointsElement.getScalarByPropertyUri(AML_REST_SDK_PAGINATION_TYPE);

    return pagingTypes.isEmpty() ? null : pagingTypes.get(0).toString();
  }

  private ExpressionDescriptor parseResponseValueExtractionDescriptor(List<DialectDomainElement> valueExtraction) {
    if (!valueExtraction.isEmpty()) {
      return expressionParser.parseExpression(valueExtraction.get(0));
    }

    return null;
  }
}
