/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.descriptor.writer;

import static java.lang.System.lineSeparator;
import static java.util.regex.Pattern.compile;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import java.util.regex.Pattern;

import javax.validation.constraints.NotNull;

public class YamlWriter {

  private final static String INDENTATION_VALUE = " ";
  private final static int INDENTATION_MULTIPLIER = 2;

  private final static String COMMENT_VALUE = "#";
  private final static String QUOTE_VALUE = "'";
  private final static String ESCAPED_QUOTE_VALUE = "''";
  private final static String WHITE_SPACE = " ";

  private static final Pattern NEEDS_ESCAPING_PATTERN = compile("[\\[\\]\\#\\:\\{\\}\\\"\\'\\>\\?\\@]");

  private final StringBuilder stringBuilder = new StringBuilder();
  private int indentationLevel = 0;

  public YamlWriter addIndentation() {
    this.indentationLevel++;
    return this;
  }

  public YamlWriter removeIndentation() {
    this.indentationLevel--;

    if (indentationLevel < 0) {
      indentationLevel = 0;
    }

    return this;
  }

  public YamlWriter addKeyLine(@NotNull String key) {
    return addKeyLine(key, null);
  }

  public YamlWriter addKeyLine(@NotNull String key, String comment) {
    return addLine(buildKeyValueCommentLine(key, null, comment));
  }

  public YamlWriter addKeyValueLineIfNotEmpty(@NotNull String key, String value) {
    if (isNotBlank(value)) {
      return addKeyValueLine(key, value, null);
    }
    return this;
  }

  public YamlWriter addKeyValueLine(@NotNull String key, String value) {
    return addKeyValueLine(key, value, null);
  }

  public YamlWriter addKeyValueLine(@NotNull String key, String value, String comment) {
    return addLine(buildKeyValueCommentLine(key, value, comment));
  }

  private String buildKeyValueCommentLine(@NotNull String key, String value, String comment) {
    String line = key + ":";

    if (isNotBlank(value)) {
      line += WHITE_SPACE + buildValueString(value);
    }

    if (isNotBlank(comment)) {
      line += WHITE_SPACE + COMMENT_VALUE + comment;
    }
    return line;
  }

  private String buildValueString(String value) {
    if (value.contains(lineSeparator())) {
      StringBuilder valueLine = new StringBuilder("|");
      valueLine.append(lineSeparator());
      addIndentation();
      String[] lines = value.split(lineSeparator());
      for (String line : lines) {
        if (!line.isEmpty()) {
          valueLine
              .append(getIndentationString())
              .append(line.replaceAll("^\\s+", ""));
        }
        valueLine.append(lineSeparator());
      }
      valueLine.delete(valueLine.lastIndexOf(lineSeparator()), valueLine.length());
      removeIndentation();
      return valueLine.toString();
    } else {
      return escapeValue(value);
    }
  }

  private String escapeValue(String value) {
    if (NEEDS_ESCAPING_PATTERN.matcher(value).find()) {
      return QUOTE_VALUE + value.replace(QUOTE_VALUE, ESCAPED_QUOTE_VALUE) + QUOTE_VALUE;
    }
    return value;
  }

  public YamlWriter addLine(String line) {
    stringBuilder
        .append(getIndentationString())
        .append(line)
        .append(lineSeparator());
    return this;
  }

  private String getIndentationString() {
    StringBuilder indentationString = new StringBuilder();
    for (int e = 0; e < indentationLevel; e++) {
      for (int i = 0; i < INDENTATION_MULTIPLIER; i++) {
        indentationString.append(INDENTATION_VALUE);
      }
    }
    return indentationString.toString();
  }

  @Override
  public String toString() {
    return stringBuilder.toString();
  }

}
