/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.templating;

import static java.lang.Integer.MAX_VALUE;
import static java.nio.charset.StandardCharsets.UTF_8;
import static java.nio.file.Files.delete;
import static java.nio.file.Files.find;
import static java.nio.file.Files.readAllBytes;
import static java.nio.file.Files.write;
import static java.nio.file.StandardCopyOption.REPLACE_EXISTING;
import static java.util.stream.Collectors.toList;
import static com.mulesoft.connectivity.rest.sdk.internal.util.JavaUtils.getJavaUpperCamelNameFromXml;

import com.mulesoft.connectivity.rest.sdk.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.exception.TemplatingException;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.templating.sdk.SdkConnector;
import com.mulesoft.connectivity.rest.sdk.internal.templating.sdk.SdkGitIgnore;
import com.mulesoft.connectivity.rest.sdk.internal.templating.sdk.SdkPom;
import com.mulesoft.connectivity.rest.sdk.internal.templating.sdk.SdkTestLog4j;

import com.google.googlejavaformat.java.Formatter;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;
import java.util.stream.Stream;

public class SdkConnectorTemplateEntity extends TemplateEntity {

  private static final Formatter FORMATTER = new Formatter();
  private final SdkConnector sdkConnector;
  private final SdkPom sdkPom;
  private final SdkTestLog4j sdkTestLog4j;
  private final Path projectDir;
  private final Path descriptorPath;
  private final SdkGitIgnore sdkGitIgnore;
  private final RestSdkRunConfiguration runConfiguration;

  public SdkConnectorTemplateEntity(Path outputDir, ConnectorModel connectorModel, Path descriptorPath,
                                    RestSdkRunConfiguration runConfiguration)
      throws TemplatingException {
    String projectName = getJavaUpperCamelNameFromXml(connectorModel.getConnectorXmlName());
    projectDir = outputDir.resolve(projectName);

    this.sdkConnector = new SdkConnector(projectDir, connectorModel, runConfiguration);
    this.sdkPom = new SdkPom(projectDir, connectorModel);
    this.descriptorPath = descriptorPath;
    this.sdkGitIgnore = new SdkGitIgnore(projectDir);
    this.sdkTestLog4j = new SdkTestLog4j(projectDir);
    this.runConfiguration = runConfiguration;
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    sdkConnector.applyTemplates();
    sdkPom.applyTemplates();
    sdkGitIgnore.applyTemplates();
    sdkTestLog4j.applyTemplates();
    formatJavaFiles();
    addDescriptorToSources();
  }

  protected void addDescriptorToSources() throws TemplatingException {
    if (descriptorPath != null) {
      try (InputStream descriptorStream = new FileInputStream(descriptorPath.toFile())) {
        Files.copy(descriptorStream,
                   sdkConnector.getSourcesPath().resolve("connector-descriptor.yaml"),
                   REPLACE_EXISTING);
      } catch (IOException e) {
        throw new TemplatingException("Could not add descriptor to generated sources");
      }
    }
  }

  private void formatJavaFiles() throws TemplatingException {
    for (Path path : getJavaFiles()) {
      javaFormat(path);
    }
  }

  private List<Path> getJavaFiles() throws TemplatingException {
    try (Stream<Path> paths = find(projectDir,
                                   MAX_VALUE,
                                   (filePath, fileAttr) -> filePath.toString().toLowerCase().endsWith(".java"))) {
      return paths.collect(toList());
    } catch (Exception e) {
      throw new TemplatingException("Could not find files to format", e);
    }
  }

  private void javaFormat(Path filepath) throws TemplatingException {
    String formattedSource;

    try {
      String content = new String(readAllBytes(filepath), UTF_8);
      formattedSource = FORMATTER.formatSource(content);

      delete(filepath);
      write(filepath, formattedSource.getBytes(UTF_8));
    } catch (Exception e) {
      throw new TemplatingException("Could not format file " + filepath.toString(), e);
    }
  }
}
