/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.templating.sdk;

import static javax.lang.model.element.Modifier.PRIVATE;
import static javax.lang.model.element.Modifier.PROTECTED;
import static javax.lang.model.element.Modifier.PUBLIC;

import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.http.api.HttpConstants;
import org.mule.sdk.api.annotation.param.Parameter;

import com.mulesoft.connectivity.rest.commons.api.data.sample.RestSampleDataProvider;
import com.mulesoft.connectivity.rest.commons.api.source.RequestParameterBinding;
import com.mulesoft.connectivity.rest.commons.internal.util.RestRequestBuilder;
import com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils;
import com.mulesoft.connectivity.rest.sdk.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.exception.TemplatingException;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.templating.JavaTemplateEntity;

import java.nio.file.Path;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.JavaFile;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeSpec;
import com.squareup.javapoet.WildcardTypeName;

public class SdkSampleDataProvider extends JavaTemplateEntity {

  private static final String GET_TYPED_VALUE_OR_NULL_METHOD = "getTypedValueOrNull";
  private static final String GET_PATH_TEMPLATE_METHOD = "getPathTemplate";
  private static final String GET_REQUEST_BUILDER_METHOD = "getRequestBuilder";
  private static final String GET_PARAMETER_BINDING_METHOD = "getParameterBinding";
  private static final String PARAMETER_BINDING_LOCAL_VARIABLE = "binding";
  private static final String ADD_URI_BINDING_METHOD = "addUriParamBinding";
  private static final String ADD_QUERY_BINDING_METHOD = "addQueryParamBinding";
  private static final String ADD_HEADER_BINDING_METHOD = "addHeaderBinding";
  private static final String GET_PARAMETER_VALUES_METHOD = "getParameterValues";

  private static final String PATH_TEMPLATE_FIELD = "PATH";


  private final AbstractSdkOperation sdkOperation;
  private final ConnectorOperation operation;
  private final String javaClassName;
  private final String aPackage;
  private final Class outputType;

  private Map<String, SdkParameter> operationParameters = null;

  public SdkSampleDataProvider(Path outputDir, ConnectorModel connectorModel, AbstractSdkOperation sdkOperation,
                               ConnectorOperation operation,
                               RestSdkRunConfiguration runConfiguration) {
    super(outputDir, connectorModel, runConfiguration);

    this.sdkOperation = sdkOperation;
    this.operation = operation;

    javaClassName = sdkOperation.getJavaClassName();
    aPackage = sdkOperation.getPackage();
    outputType = this.operation.hasPagination() ? String.class : Void.class;
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    generateSampleDataClass();
  }

  public String getPackage() {
    return aPackage + ".sample.data";
  }

  public String getJavaClassName() {
    return javaClassName + "SampleDataProvider";
  }

  private void generateSampleDataClass() throws TemplatingException {
    TypeSpec.Builder sampleDataClassBuilder =
        TypeSpec
            .classBuilder(getJavaClassName())
            .addModifiers(PUBLIC)
            .superclass(getSuperclass())
            .addMethod(generateConstructor())
            .addMethod(generateGetPathTemplateMethod())
            .addMethod(generateGetRequestBuilderMethod())
            .addMethod(generateGetParameterBindingMethod())
            .addMethod(generateGetParameterValuesMethod());

    addClassConstants(sampleDataClassBuilder);
    addParameters(sampleDataClassBuilder);

    JavaFile.Builder javaFileBuilder = getJavaFileBuilderForClass(sampleDataClassBuilder.build(), getPackage());
    javaFileBuilder.addStaticImport(RestSdkUtils.class, GET_TYPED_VALUE_OR_NULL_METHOD);
    writeJavaFile(javaFileBuilder.build());
  }

  private ParameterizedTypeName getSuperclass() {
    Class outputType = this.outputType;
    Class attributeOutputType = Void.class;
    return ParameterizedTypeName.get(RestSampleDataProvider.class, outputType, attributeOutputType);
  }

  private MethodSpec generateConstructor() {
    MethodSpec.Builder constructorBuilder =
        MethodSpec.constructorBuilder()
            .addModifiers(PUBLIC);

    return constructorBuilder.build();
  }

  private MethodSpec generateGetPathTemplateMethod() {
    MethodSpec.Builder methodBuilder =
        MethodSpec.methodBuilder(GET_PATH_TEMPLATE_METHOD)
            .addModifiers(PROTECTED)
            .returns(String.class)
            .addAnnotation(Override.class);

    methodBuilder.addStatement("return $L", PATH_TEMPLATE_FIELD);

    return methodBuilder.build();
  }

  private MethodSpec generateGetRequestBuilderMethod() {
    MethodSpec.Builder methodBuilder =
        MethodSpec.methodBuilder(GET_REQUEST_BUILDER_METHOD)
            .addModifiers(PROTECTED)
            .returns(RestRequestBuilder.class)
            .addAnnotation(Override.class)
            .addParameter(String.class, "path");

    methodBuilder.addStatement("return new $T(connection.getBaseUri(), path, $T.$L)",
                               RestRequestBuilder.class,
                               HttpConstants.Method.class,
                               operation.getHttpMethod()
                                   .toUpperCase());

    return methodBuilder.build();
  }

  private MethodSpec generateGetParameterBindingMethod() {
    MethodSpec.Builder methodBuilder =
        MethodSpec.methodBuilder(GET_PARAMETER_BINDING_METHOD)
            .addModifiers(PROTECTED)
            .returns(RequestParameterBinding.class)
            .addAnnotation(Override.class);

    Map<String, SdkParameter> operationValueResolverParameters = getOperationValueResolverParameters();


    if (!operationValueResolverParameters.isEmpty()) {
      methodBuilder.addStatement("$1T $2L = new $1T()",
                                 RequestParameterBinding.class,
                                 PARAMETER_BINDING_LOCAL_VARIABLE);
      addParameter(methodBuilder, sdkOperation.allQueryParameters, ADD_QUERY_BINDING_METHOD);
      addParameter(methodBuilder, sdkOperation.allPathParameters, ADD_URI_BINDING_METHOD);
      addParameter(methodBuilder, sdkOperation.allHeaders, ADD_HEADER_BINDING_METHOD);
      methodBuilder.addStatement("return $1L", PARAMETER_BINDING_LOCAL_VARIABLE);
    } else {
      methodBuilder.addStatement("return new $1T()", RequestParameterBinding.class);
    }

    return methodBuilder.build();
  }

  private void addParameter(MethodSpec.Builder methodBuilder, List<SdkParameter> parameters, String bindingMethod) {
    for (SdkParameter sdkParameter : parameters) {
      methodBuilder.addStatement("$L.$L($S, $L)",
                                 PARAMETER_BINDING_LOCAL_VARIABLE,
                                 bindingMethod,
                                 sdkParameter.getExternalName(),
                                 sdkParameter.getStringValueGetter());
    }
  }

  private MethodSpec generateGetParameterValuesMethod() {
    MethodSpec.Builder methodBuilder =
        MethodSpec.methodBuilder(GET_PARAMETER_VALUES_METHOD)
            .addModifiers(PROTECTED)
            .returns(getParameterValuesMultiMapType())
            .addAnnotation(Override.class);
    methodBuilder.addStatement("return new $T()", getParameterValuesMultiMapType());
    return methodBuilder.build();
  }

  private Map<String, SdkParameter> getOperationValueResolverParameters() {
    if (operationParameters != null) {
      return operationParameters;
    }

    final Map<String, SdkParameter> operationParameters = new HashMap<>();
    for (SdkParameter sdkParameter : sdkOperation.allQueryParameters) {
      operationParameters.put(sdkParameter.getJavaName(), sdkParameter);
    }
    for (SdkParameter sdkParameter : sdkOperation.allPathParameters) {
      operationParameters.put(sdkParameter.getJavaName(), sdkParameter);
    }
    for (SdkParameter sdkParameter : sdkOperation.allHeaders) {
      operationParameters.put(sdkParameter.getJavaName(), sdkParameter);
    }
    this.operationParameters = operationParameters;
    return this.operationParameters;
  }

  private ParameterizedTypeName getParameterValuesMultiMapType() {
    ParameterizedTypeName wildcardTypedValueType =
        ParameterizedTypeName.get(
                                  ClassName.get(TypedValue.class),
                                  WildcardTypeName.subtypeOf(Object.class));

    return ParameterizedTypeName.get(ClassName.get(MultiMap.class), ClassName.get(String.class), wildcardTypedValueType);
  }

  private void addClassConstants(TypeSpec.Builder sampleDataClassBuilder) {
    sampleDataClassBuilder
        .addField(getConstantStringField(PATH_TEMPLATE_FIELD, operation.getPath()));
  }

  private void addParameters(TypeSpec.Builder valueResolverClassBuilder) {
    Map<String, SdkParameter> operationSdkParameters = getOperationValueResolverParameters();

    for (String parameterName : operationSdkParameters.keySet()) {
      valueResolverClassBuilder.addField(getParameterFieldSpec(operationSdkParameters.get(parameterName)));
    }
  }

  private FieldSpec getParameterFieldSpec(SdkParameter sdkParameter) {
    FieldSpec.Builder fieldBuilder =
        FieldSpec.builder(sdkParameter.getTypeName(),
                          sdkParameter.getJavaName(),
                          PRIVATE)
            .addAnnotation(Parameter.class);
    return fieldBuilder.build();
  }
}
