/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.templating.sdk;

import static com.mulesoft.connectivity.rest.sdk.internal.util.JavaUtils.getJavaLowerCamelNameFromXml;
import static javax.lang.model.element.Modifier.PRIVATE;
import static javax.lang.model.element.Modifier.PROTECTED;
import static javax.lang.model.element.Modifier.PUBLIC;

import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.http.api.HttpConstants;
import org.mule.sdk.api.annotation.param.Parameter;

import com.mulesoft.connectivity.rest.commons.api.data.sample.RestSampleDataProviderTrigger;
import com.mulesoft.connectivity.rest.commons.api.source.RequestParameterBinding;
import com.mulesoft.connectivity.rest.commons.internal.util.RestRequestBuilder;
import com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils;
import com.mulesoft.connectivity.rest.sdk.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.exception.TemplatingException;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.trigger.ParameterBinding;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.trigger.Trigger;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.trigger.TriggerParameter;
import com.mulesoft.connectivity.rest.sdk.internal.templating.JavaTemplateEntity;

import java.nio.file.Path;

import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.JavaFile;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeSpec;
import com.squareup.javapoet.WildcardTypeName;

public class SdkSampleDataProviderTrigger extends JavaTemplateEntity {

  private static final String GET_TYPED_VALUE_OR_NULL_METHOD = "getTypedValueOrNull";
  private static final String GET_PATH_TEMPLATE_METHOD = "getPathTemplate";
  private static final String GET_REQUEST_BUILDER_METHOD = "getRequestBuilder";
  private static final String GET_PARAMETER_BINDING_METHOD = "getParameterBinding";
  private static final String PARAMETER_BINDING_LOCAL_VARIABLE = "binding";
  private static final String ADD_URI_BINDING_METHOD = "addUriParamBinding";
  private static final String ADD_QUERY_BINDING_METHOD = "addQueryParamBinding";
  private static final String ADD_HEADER_BINDING_METHOD = "addHeaderBinding";
  private static final String GET_PARAMETER_VALUES_METHOD = "getParameterValues";

  private static final String PATH_TEMPLATE_FIELD = "PATH";
  private static final String ITEMS_EXPRESSION_FIELD = "ITEMS_EXPRESSION";
  private static final String PARAMETER_VALUES_LOCAL_VARIABLE = "parameterValues";


  private final Trigger trigger;
  private final ConnectorOperation operation;
  private final String javaClassName;
  private final String aPackage;

  public SdkSampleDataProviderTrigger(Path outputDir, ConnectorModel connectorModel,
                                      SdkTrigger sdkTrigger, Trigger trigger,
                                      RestSdkRunConfiguration runConfiguration) {
    super(outputDir, connectorModel, runConfiguration);

    this.trigger = trigger;
    this.operation = trigger.getOperation();
    javaClassName = sdkTrigger.getJavaClassName();
    aPackage = sdkTrigger.getPackage();
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    generateValueProviderClass();
  }

  public String getPackage() {
    return aPackage + ".sample.data";
  }

  public String getJavaClassName() {
    return javaClassName + "SampleDataProvider";
  }

  private void generateValueProviderClass() throws TemplatingException {
    TypeSpec.Builder valueProviderClassBuilder =
        TypeSpec
            .classBuilder(getJavaClassName())
            .addModifiers(PUBLIC)
            .superclass(getSuperclass())
            .addMethod(generateConstructor())
            .addMethod(generateGetPathTemplateMethod())
            .addMethod(generateGetRequestBuilderMethod())
            .addMethod(generateGetParameterBindingMethod())
            .addMethod(generateGetParameterValuesMethod());

    addClassConstants(valueProviderClassBuilder);
    addParameters(valueProviderClassBuilder);

    JavaFile.Builder javaFileBuilder = getJavaFileBuilderForClass(valueProviderClassBuilder.build(), getPackage());
    javaFileBuilder.addStaticImport(RestSdkUtils.class, GET_TYPED_VALUE_OR_NULL_METHOD);
    writeJavaFile(javaFileBuilder.build());
  }

  private Class getSuperclass() {
    return RestSampleDataProviderTrigger.class;
  }

  private MethodSpec generateConstructor() {
    MethodSpec.Builder constructorBuilder =
        MethodSpec.constructorBuilder()
            .addModifiers(PUBLIC);

    constructorBuilder.addStatement("super($L)",
                                    ITEMS_EXPRESSION_FIELD);
    return constructorBuilder.build();
  }

  private MethodSpec generateGetPathTemplateMethod() {
    MethodSpec.Builder methodBuilder =
        MethodSpec.methodBuilder(GET_PATH_TEMPLATE_METHOD)
            .addModifiers(PROTECTED)
            .returns(String.class)
            .addAnnotation(Override.class);

    methodBuilder.addStatement("return $L", PATH_TEMPLATE_FIELD);

    return methodBuilder.build();
  }

  private MethodSpec generateGetRequestBuilderMethod() {
    MethodSpec.Builder methodBuilder =
        MethodSpec.methodBuilder(GET_REQUEST_BUILDER_METHOD)
            .addModifiers(PROTECTED)
            .returns(RestRequestBuilder.class)
            .addAnnotation(Override.class)
            .addParameter(String.class, "path");

    methodBuilder.addStatement("return new $T(connection.getBaseUri(), path, $T.$L)",
                               RestRequestBuilder.class,
                               HttpConstants.Method.class,
                               operation.getHttpMethod()
                                   .toUpperCase());

    return methodBuilder.build();
  }

  private MethodSpec generateGetParameterBindingMethod() {
    MethodSpec.Builder methodBuilder =
        MethodSpec.methodBuilder(GET_PARAMETER_BINDING_METHOD)
            .addModifiers(PROTECTED)
            .returns(RequestParameterBinding.class)
            .addAnnotation(Override.class);

    methodBuilder.addStatement("$1T $2L = new $1T()",
                               RequestParameterBinding.class,
                               PARAMETER_BINDING_LOCAL_VARIABLE);

    if (trigger.getParameterBindings() != null) {
      for (ParameterBinding binding : trigger.getParameterBindings()) {
        methodBuilder.addStatement("$L.$L($S, $S)",
                                   PARAMETER_BINDING_LOCAL_VARIABLE,
                                   getParameterBindingAddMethodName(binding),
                                   binding.getName(),
                                   binding.getExpression());
      }
    }

    methodBuilder.addStatement("return $1L", PARAMETER_BINDING_LOCAL_VARIABLE);

    return methodBuilder.build();
  }

  private String getParameterBindingAddMethodName(ParameterBinding binding) {
    switch (binding.getParameterType()) {
      case QUERY:
        return ADD_QUERY_BINDING_METHOD;
      case HEADER:
        return ADD_HEADER_BINDING_METHOD;
      case URI:
        return ADD_URI_BINDING_METHOD;
    }

    throw new IllegalArgumentException("Parameter type not supported: " + binding.getParameterType());
  }

  private MethodSpec generateGetParameterValuesMethod() {
    MethodSpec.Builder methodBuilder =
        MethodSpec.methodBuilder(GET_PARAMETER_VALUES_METHOD)
            .addModifiers(PROTECTED)
            .returns(getParameterValuesMultiMapType())
            .addAnnotation(Override.class);

    methodBuilder.addStatement("final $T $L = new $T<>()",
                               getParameterValuesMultiMapType(),
                               PARAMETER_VALUES_LOCAL_VARIABLE,
                               MultiMap.class);

    for (TriggerParameter parameter : trigger.getParameters()) {
      methodBuilder.addStatement("$L.put($S, $L($L))",
                                 PARAMETER_VALUES_LOCAL_VARIABLE,
                                 parameter.getExternalName(),
                                 GET_TYPED_VALUE_OR_NULL_METHOD,
                                 getJavaLowerCamelNameFromXml(parameter.getInternalName()));
    }

    methodBuilder.addStatement("return $L", PARAMETER_VALUES_LOCAL_VARIABLE);

    return methodBuilder.build();
  }

  private ParameterizedTypeName getParameterValuesMultiMapType() {
    ParameterizedTypeName wildcardTypedValueType =
        ParameterizedTypeName.get(
                                  ClassName.get(TypedValue.class),
                                  WildcardTypeName.subtypeOf(Object.class));

    return ParameterizedTypeName.get(ClassName.get(MultiMap.class), ClassName.get(String.class), wildcardTypedValueType);
  }

  private void addClassConstants(TypeSpec.Builder valueProviderClassBuilder) {
    valueProviderClassBuilder
        .addField(getConstantStringField(PATH_TEMPLATE_FIELD, operation.getPath()))
        .addField(getConstantStringField(ITEMS_EXPRESSION_FIELD, trigger.getItemsExpression()));
  }

  private void addParameters(TypeSpec.Builder triggerClassBuilder) {
    for (TriggerParameter parameter : trigger.getParameters()) {
      triggerClassBuilder.addField(getParameter(parameter));
    }
  }

  private FieldSpec getParameter(TriggerParameter parameter) {
    FieldSpec.Builder fieldBuilder =
        FieldSpec.builder(getJavaType(parameter.getParameterType()),
                          getJavaLowerCamelNameFromXml(parameter.getInternalName()),
                          PRIVATE)
            .addAnnotation(Parameter.class);
    return fieldBuilder.build();
  }
}
