/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.util;

import com.mulesoft.connectivity.rest.sdk.exception.ModelGenerationException;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.util.Objects;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Semaphore;

import javax.validation.constraints.NotNull;

import amf.MessageStyle;
import amf.ProfileName;
import amf.client.AMF;
import amf.client.model.document.DialectInstance;
import amf.client.parse.Aml10Parser;
import amf.client.resolve.Resolver;
import amf.client.validate.ValidationReport;

import static com.google.common.io.Files.createTempDir;
import static java.lang.Thread.currentThread;
import static org.apache.commons.io.FileUtils.copyInputStreamToFile;

public final class AMFWrapper {

  private static final Semaphore initializationSemaphore = new Semaphore(1);
  private static boolean amfInitialized = false;

  private static final String REST_SDK_AML_RESOURCES_PATH = "com/mulesoft/connectivity/rest/aml";
  private static final String DIALECTS_FOLDER = "/dialects";
  private static final String VOCABULARIES_FOLDER = "/vocabularies";
  private static final String REST_SDK_VOCABULARY_FILE_NAME = "rest_sdk_vocabulary.yaml";
  private static final String REST_SDK_DIALECT_FILE_NAME = "rest_sdk_dialect.yaml";

  private static final String DIALECT_RESOURCE_PATH =
      REST_SDK_AML_RESOURCES_PATH + DIALECTS_FOLDER + "/" + REST_SDK_DIALECT_FILE_NAME;
  private static final String VOCABULARY_RESOURCE_PATH =
      REST_SDK_AML_RESOURCES_PATH + VOCABULARIES_FOLDER + "/" + REST_SDK_VOCABULARY_FILE_NAME;


  private AMFWrapper() {}

  public static void initialize() throws ModelGenerationException {
    if (!amfInitialized) {
      try {
        initializationSemaphore.acquire();
        if (!amfInitialized) {
          AMF.init().get();
          registerDialect();
        }
      } catch (ExecutionException e) {
        throw new ModelGenerationException("Error initializing AMF.", e);
      } catch (InterruptedException e) {
        currentThread().interrupt();
        throw new ModelGenerationException("Error initializing AMF.", e);
      } finally {
        initializationSemaphore.release();
      }
      amfInitialized = true;
    }
  }

  private static void registerDialect() throws ModelGenerationException {
    try {
      InputStream source = currentThread().getContextClassLoader().getResourceAsStream(DIALECT_RESOURCE_PATH);
      InputStream vocabulary = currentThread().getContextClassLoader().getResourceAsStream(VOCABULARY_RESOURCE_PATH);

      Objects.requireNonNull(source);
      Objects.requireNonNull(vocabulary);

      File baseTmpDir = createTempDir();

      File baseDir = new File(baseTmpDir.toPath() + DIALECTS_FOLDER);
      File baseDirVoc = new File(baseTmpDir.toPath() + VOCABULARIES_FOLDER);
      File tmpDialect = new File(baseDir, REST_SDK_DIALECT_FILE_NAME);
      File tempVocabulary = new File(baseDirVoc, REST_SDK_VOCABULARY_FILE_NAME);
      copyInputStreamToFile(vocabulary, tempVocabulary);
      copyInputStreamToFile(source, tmpDialect);

      AMF.registerDialect(tmpDialect.toURI().toURL().toString()).get();
    } catch (ExecutionException | IOException e) {
      throw new ModelGenerationException("Could not register dialect.", e);
    } catch (InterruptedException e) {
      currentThread().interrupt();
      throw new ModelGenerationException("Could not register dialect.", e);
    }
  }

  public static DialectInstance parseConnectorDescriptor(@NotNull File connectorDescriptor)
      throws ModelGenerationException {

    initialize();

    DialectInstance dialectInstance;
    try {
      dialectInstance = (DialectInstance) new Aml10Parser().parseFileAsync(connectorDescriptor.toURI().toURL().toString()).get();
    } catch (ExecutionException | MalformedURLException e) {
      throw new ModelGenerationException("Could not parse connector descriptor.", e);
    } catch (InterruptedException e) {
      currentThread().interrupt();
      throw new ModelGenerationException("Could not parse connector descriptor.", e);
    }

    dialectInstance = (DialectInstance) new Resolver("AML 1.0").resolve(dialectInstance);
    validate(dialectInstance);
    return Objects.requireNonNull(dialectInstance);
  }

  public static void validate(DialectInstance metadata)
      throws ModelGenerationException {

    ValidationReport report;
    try {
      report =
          AMF.validate(metadata, ProfileName.apply("Rest Connector Descriptor 1.0"), MessageStyle.apply("AMF")).get();
    } catch (Exception e) {
      currentThread().interrupt();
      throw new ModelGenerationException("Could not validate connector descriptor", e);
    }

    if (!report.conforms()) {
      throw new ModelGenerationException(report.toString());
    }
  }
}
