/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules;

import static java.util.stream.Collectors.toList;
import static com.mulesoft.connectivity.rest.sdk.internal.util.NamingUtil.isFriendlyName;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationRule.Level.WARN;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.EndPointDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.PreValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.LinkedList;
import java.util.List;

public class ParameterNameMustBeFriendlyRule extends PreValidationRule {

  //R013
  public ParameterNameMustBeFriendlyRule() {
    super("Parameter display name seems not to be a friendly name",
          "The displayName should be a human readable name separated with spaces.",
          WARN);
  }

  @Override
  public List<ValidationResult> preValidate(ConnectorDescriptor connectorDescriptor, APIModel apiModel) {

    List<ValidationResult> results = new LinkedList<>();

    for (EndPointDescriptor endpointDescriptor : connectorDescriptor.getEndpoints()) {
      for (OperationDescriptor operationDescriptor : endpointDescriptor.getOperations()) {
        if (operationDescriptor.getExpects() != null) {
          results.addAll(validateParameterNames(operationDescriptor.getExpects().getUriParameter(), ParameterType.URI,
                                                operationDescriptor,
                                                endpointDescriptor));
          results.addAll(validateParameterNames(operationDescriptor.getExpects().getQueryParameter(), ParameterType.QUERY,
                                                operationDescriptor,
                                                endpointDescriptor));
          results.addAll(validateParameterNames(operationDescriptor.getExpects().getHeader(), ParameterType.HEADER,
                                                operationDescriptor,
                                                endpointDescriptor));
        }
      }
    }

    return results;
  }

  List<ValidationResult> validateParameterNames(List<ParameterDescriptor> parameters, ParameterType parameterType,
                                                OperationDescriptor operationDescriptor, EndPointDescriptor endpointDescriptor) {
    return parameters.stream()
        .filter(x -> !isFriendlyName(x.getFriendlyName()))
        .map(x -> getValidationError(endpointDescriptor, operationDescriptor, x, parameterType))
        .collect(toList());
  }

  private ValidationResult getValidationError(EndPointDescriptor endpointDescriptor, OperationDescriptor operationDescriptor,
                                              ParameterDescriptor parameterDescriptor, ParameterType parameterType) {
    String location =
        "Operation Descriptor with PATH: " + endpointDescriptor.getPath() +
            " and METHOD: " + operationDescriptor.getMethod() +
            " declares a " + parameterType.getName() + " named " + parameterDescriptor.getFriendlyName() +
            " whose display name is : " + parameterDescriptor.getFriendlyName();

    return new ValidationResult(this, location);
  }
}
