/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules;

import static org.apache.commons.lang3.StringUtils.EMPTY;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationRule.Level.ERROR;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TriggerDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.PreValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIParameterModel;

import java.util.LinkedList;
import java.util.List;

public class TriggerUriParamsMustBeCompleteRule extends PreValidationRule {

  public TriggerUriParamsMustBeCompleteRule() {
    super("All URI parameters must be have a binding in the Trigger Descriptor.", EMPTY, ERROR);
  }

  @Override
  public List<ValidationResult> preValidate(ConnectorDescriptor connectorDescriptor, APIModel apiModel) {
    List<ValidationResult> results = new LinkedList<>();

    List<TriggerDescriptor> triggerDescriptors = connectorDescriptor.getTriggers();
    if (!triggerDescriptors.isEmpty()) {
      for (TriggerDescriptor triggerDescriptor : connectorDescriptor.getTriggers()) {

        APIOperationModel operationModel = getOperation(apiModel, triggerDescriptor);
        if (operationModel == null) {
          continue;
        }

        List<APIParameterModel> uriParamsModel = operationModel.getUriParamsModel();
        if (uriParamsModel.size() > triggerDescriptor.getParameterBindings().getUriParameters().size()) {
          results.add(getValidationError(triggerDescriptor));
        }
      }
    }

    return results;
  }

  private APIOperationModel getOperation(APIModel apiModel, TriggerDescriptor triggerDescriptor) {
    return apiModel.getOperationsModel().stream()
        .filter(x -> x.getPath().equalsIgnoreCase(triggerDescriptor.getPath()))
        .filter(x -> x.getHttpMethod().name().equalsIgnoreCase(triggerDescriptor.getMethod().getName()))
        .findFirst().orElse(null);
  }

  private ValidationResult getValidationError(TriggerDescriptor triggerDescriptor) {
    String location =
        "Trigger with PATH: " + triggerDescriptor.getPath() + " and METHOD: " + triggerDescriptor.getMethod().getName();
    return new ValidationResult(this, location);
  }
}
