/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationRule.Level.ERROR;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isBlank;

import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.EndPointDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ValueResolverDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ValueResolverReferenceArgumentDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.PreValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;

import java.util.LinkedList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class ValueResolverReferenceArgumentsOperationParameterOnlySentOnce extends PreValidationRule {

  public ValueResolverReferenceArgumentsOperationParameterOnlySentOnce() {
    super("Reference to operation parameters used as arguments for a value resolver can only be used once.", EMPTY, ERROR);
  }

  @Override
  public List<ValidationResult> preValidate(ConnectorDescriptor connectorDescriptor, APIModel apiModel) {
    final List<ValidationResult> validationResults = new LinkedList<>();

    for (EndPointDescriptor endpoint : connectorDescriptor.getEndpoints()) {
      for (OperationDescriptor operation : endpoint.getOperations()) {
        validationResults
            .addAll(validateOperationParameters(endpoint, operation, operation.getExpects().getUriParameter(), apiModel,
                                                connectorDescriptor.getValueResolvers()));
        validationResults
            .addAll(validateOperationParameters(endpoint, operation, operation.getExpects().getQueryParameter(), apiModel,
                                                connectorDescriptor.getValueResolvers()));
        validationResults
            .addAll(validateOperationParameters(endpoint, operation, operation.getExpects().getHeader(), apiModel,
                                                connectorDescriptor.getValueResolvers()));
      }
    }
    return validationResults;
  }

  private List<ValidationResult> validateOperationParameters(EndPointDescriptor endpointDescriptor,
                                                             OperationDescriptor operationDescriptor,
                                                             List<ParameterDescriptor> parameterDescriptors,
                                                             APIModel apiModel,
                                                             List<ValueResolverDescriptor> valueResolverDescriptors) {
    final List<ValidationResult> validationResults = new LinkedList<>();

    for (ParameterDescriptor parameter : parameterDescriptors) {
      if (parameter.getValueResolver() != null
          && parameter.getValueResolver().getArguments() != null
          && !parameter.getValueResolver().getArguments().isEmpty()) {

        if (valueResolverDescriptors.isEmpty()
            || valueResolverDescriptors.stream()
                .noneMatch(x -> x.getName().equalsIgnoreCase(parameter.getValueResolver().getId()))) {
          continue;
        }

        final APIOperationModel apiOperationModel = findApiOperationModel(apiModel, endpointDescriptor, operationDescriptor);

        final List<String> usedUriParams = new LinkedList<>();
        final List<String> usedQueryParams = new LinkedList<>();
        final List<String> usedHeaders = new LinkedList<>();

        if (apiOperationModel != null) {
          validationResults.addAll(
                                   parameter.getValueResolver().getArguments().stream()
                                       .filter(x -> validArgumentValueFormat(x.getValue()))
                                       .filter(x -> isDuplicated(x, usedUriParams, usedQueryParams, usedHeaders))
                                       .map(x -> getValidationError(endpointDescriptor, operationDescriptor, parameter, x))
                                       .collect(toList()));
        }
      }
    }

    return validationResults;
  }

  private boolean isDuplicated(ValueResolverReferenceArgumentDescriptor argument, List<String> usedUriParams,
                               List<String> usedQueryParams, List<String> usedHeaders) {
    final String parameterType = getParameterTypeReference(argument.getValue());
    final String parameterName = getParameterReference(argument.getValue());

    if (parameterType == null || parameterName == null) {
      return false;
    }

    if (parameterType.startsWith("uriParameter")) {
      if (usedUriParams.stream().anyMatch(x -> x.equals(parameterName))) {
        return true;
      }
      usedUriParams.add(parameterName);
    } else if (parameterType.startsWith("queryParameter")) {
      if (usedQueryParams.stream().anyMatch(x -> x.equals(parameterName))) {
        return true;
      }
      usedQueryParams.add(parameterName);
    } else if (parameterType.startsWith("header")) {
      if (usedHeaders.stream().anyMatch(x -> x.equals(parameterName))) {
        return true;
      }
      usedHeaders.add(parameterName);
    }

    return false;
  }

  private APIOperationModel findApiOperationModel(APIModel apiModel, EndPointDescriptor endpointDescriptor,
                                                  OperationDescriptor operationDescriptor) {
    return apiModel.getOperationsModel().stream()
        .filter(x -> x.getPath().equalsIgnoreCase(endpointDescriptor.getPath()))
        .filter(x -> x.getHttpMethod().name().equalsIgnoreCase(operationDescriptor.getMethod()))
        .findFirst().orElse(null);
  }

  private static final Pattern VALID_FORMAT_PATTERN = Pattern.compile("^(?:uriParameter\\.|queryParameter\\.|header\\.)\\S+$");

  private boolean validArgumentValueFormat(String value) {
    if (isBlank(value)) {
      return false;
    }

    return VALID_FORMAT_PATTERN.matcher(value).matches();
  }

  private static final Pattern PARAMETER_NAME_PATTERN = Pattern.compile("^(uriParameter\\.|queryParameter\\.|header\\.)(\\S+)$");

  private String getParameterTypeReference(String value) {
    if (isBlank(value)) {
      return null;
    }

    final Matcher matcher = PARAMETER_NAME_PATTERN.matcher(value);

    if (matcher.matches()) {
      return matcher.group(1);
    }
    return null;
  }

  private String getParameterReference(String value) {
    if (isBlank(value)) {
      return null;
    }

    final Matcher matcher = PARAMETER_NAME_PATTERN.matcher(value);

    if (matcher.matches()) {
      return matcher.group(2);
    }
    return null;
  }

  private ValidationResult getValidationError(EndPointDescriptor endPointDescriptor,
                                              OperationDescriptor operationDescriptor,
                                              ParameterDescriptor parameterDescriptor,
                                              ValueResolverReferenceArgumentDescriptor argument) {
    final String location =
        "'" + parameterDescriptor.getParamName() + "' parameter declared for the operation with PATH '"
            + endPointDescriptor.getPath()
            + "' and METHOD: '" + operationDescriptor.getMethod() + "' on the connector descriptor referencing the same operation"
            +
            " parameter more than once (For the argument '" + argument.getName() + "' : '" + argument.getValue() + "')";

    return new ValidationResult(this, location);
  }
}
