/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationRule.Level.ERROR;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;

import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.EndPointDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ValueResolverDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.PreValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.LinkedList;
import java.util.List;

public class ValueResolverReferenceExistsRule extends PreValidationRule {

  public ValueResolverReferenceExistsRule() {
    super("Value resolver reference id set for parameters must exist in the connector descriptor", EMPTY, ERROR);
  }

  @Override
  public List<ValidationResult> preValidate(ConnectorDescriptor connectorDescriptor, APIModel apiModel) {
    final List<String> valueResolverIds =
        connectorDescriptor.getValueResolvers().stream().map(ValueResolverDescriptor::getName).collect(toList());

    final List<ValidationResult> validationResults = new LinkedList<>();
    for (EndPointDescriptor endpoint : connectorDescriptor.getEndpoints()) {
      for (OperationDescriptor operation : endpoint.getOperations()) {
        if (operation.getExpects() != null) {
          validationResults
              .addAll(validateOperationParameters(valueResolverIds, endpoint, operation,
                                                  operation.getExpects().getUriParameter()));
          validationResults
              .addAll(validateOperationParameters(valueResolverIds, endpoint, operation,
                                                  operation.getExpects().getQueryParameter()));
          validationResults
              .addAll(validateOperationParameters(valueResolverIds, endpoint, operation, operation.getExpects().getHeader()));
        }
      }
    }
    return validationResults;
  }

  private List<ValidationResult> validateOperationParameters(List<String> validValueResolverIds,
                                                             EndPointDescriptor endpointDescriptor,
                                                             OperationDescriptor operationDescriptor,
                                                             List<ParameterDescriptor> parameterDescriptors) {
    if (parameterDescriptors == null) {
      return emptyList();
    }

    return parameterDescriptors.stream()
        .filter(x -> !isValidValueResolverId(validValueResolverIds, x))
        .map(x -> getValidationError(endpointDescriptor, operationDescriptor, x))
        .collect(toList());
  }

  private boolean isValidValueResolverId(List<String> validValueResolverIds, ParameterDescriptor parameter) {
    if (parameter.getValueResolver() == null) {
      return true;
    }

    if (validValueResolverIds == null || validValueResolverIds.isEmpty()) {
      return false;
    }

    if (parameter.getValueResolver().getId() == null) {
      return false;
    }

    return validValueResolverIds.stream().anyMatch(y -> parameter.getValueResolver().getId().equals(y));
  }

  private ValidationResult getValidationError(EndPointDescriptor endPointDescriptor,
                                              OperationDescriptor operationDescriptor,
                                              ParameterDescriptor parameterDescriptor) {
    final String location =
        "'" + parameterDescriptor.getParamName() + "' parameter declared for the operation with PATH '"
            + endPointDescriptor.getPath()
            + "' and METHOD: '" + operationDescriptor.getMethod() + "' on the connector descriptor is declaring an invalid" +
            " value resolver: '" + parameterDescriptor.getValueResolver().getId() + "'";

    return new ValidationResult(this, location);
  }
}
