/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.webapi.parser.amf;

import static com.mulesoft.connectivity.rest.sdk.internal.webapi.util.ParserUtils.removeHtmlTags;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.parser.amf.AMFTypeSchemaModel.isXmlSchemaShape;

import amf.client.model.domain.AnyShape;
import amf.client.model.domain.ArrayShape;
import amf.client.model.domain.FileShape;
import amf.client.model.domain.NodeShape;
import amf.client.model.domain.Parameter;
import amf.client.model.domain.Payload;
import amf.client.model.domain.PropertyShape;
import amf.client.model.domain.SchemaShape;
import amf.client.model.domain.Shape;
import amf.client.model.domain.UnionShape;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIParameterModel;

import javax.ws.rs.core.MediaType;

public class AMFParameterModel extends APIParameterModel {

  public AMFParameterModel(Parameter parameter, ParameterType parameterType, boolean isPassword) {
    this.externalName = parameter.parameterName().value();
    this.displayName = buildDisplayName(parameter);
    this.parameterType = parameterType;
    this.defaultValue = buildDefaultValue(parameter);
    this.description = buildDescription(parameter);
    this.isPassword = isPassword;
    this.isRequired = parameter.required().value();

    Shape parameterShape = getParameterShape(parameter);
    this.typeModel = new AMFTypeModel((AnyShape) parameterShape, getDefaultMediaType(parameterShape));
  }

  private String buildDisplayName(Parameter parameter) {
    Shape parameterShape = getParameterShape(parameter);

    String displayName =
        parameterShape.displayName().isNullOrEmpty() ? parameter.parameterName().value() : parameterShape.displayName().value();

    return removeHtmlTags(displayName);
  }

  private String buildDescription(Parameter parameter) {
    return removeHtmlTags(parameter.description().value());
  }

  public AMFParameterModel(PropertyShape propertyShape, ParameterType parameterType) {
    this.externalName = propertyShape.id().substring(propertyShape.id().lastIndexOf('/') + 1);
    this.displayName = propertyShape.displayName().nonEmpty() ? propertyShape.displayName().value()
        : (propertyShape.range().displayName().nonEmpty() ? propertyShape.range().displayName().value() : externalName);
    this.description = propertyShape.description().nonEmpty() ? propertyShape.description().value()
        : propertyShape.range().description().value();
    this.parameterType = parameterType;
    this.typeModel =
        new AMFTypeModel((AnyShape) propertyShape.range(), getDefaultMediaType(propertyShape.range()));
  }

  public AMFParameterModel(String name, ParameterType parameterType, AMFPrimitiveTypeModel amfPrimitiveTypeParser) {
    this.externalName = name;
    this.displayName = this.externalName;
    this.parameterType = parameterType;
    this.typeModel = new AMFTypeModel(amfPrimitiveTypeParser);
  }

  private String buildDefaultValue(Parameter parameter) {
    Shape parameterShape = getParameterShape(parameter);
    return parameterShape.defaultValueStr().nonEmpty() ? parameterShape.defaultValueStr().value() : EMPTY;
  }

  private String getDefaultMediaType(Shape shape) {
    if (shape instanceof FileShape)
      return MediaType.APPLICATION_OCTET_STREAM;

    if (shape instanceof NodeShape || shape instanceof ArrayShape || shape instanceof UnionShape)
      return MediaType.APPLICATION_JSON;

    if (shape instanceof SchemaShape) {
      if (isXmlSchemaShape(shape))
        return MediaType.APPLICATION_XML;
      else
        return MediaType.APPLICATION_JSON;
    }

    return MediaType.TEXT_PLAIN;
  }

  private static Shape getParameterShape(Parameter parameter) {
    Shape shape = parameter.schema();

    if (shape == null && parameter.payloads().size() > 0) {
      shape = parameter.payloads().stream().filter(x -> x.schema() != null).map(Payload::schema).findFirst().orElse(null);
    }

    return shape;
  }
}
