/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.webapi.parser.amf;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.ConnectorSecurityScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.BasicAuthScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.CustomAuthenticationScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.DigestAuthenticationScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.NotSupportedScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.OAuth2AuthorizationCodeScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.OAuth2ClientCredentialsScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.PassThroughScheme;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIParameterModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APISecuritySchemeModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.parser.amf.security.AMFOauth2FlowsNaming;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.parser.amf.security.AMFSecuritySchemesNaming;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.PrimitiveTypeDefinition;

import java.util.ArrayList;
import java.util.List;

import amf.client.model.domain.ApiKeySettings;
import amf.client.model.domain.HttpSettings;
import amf.client.model.domain.OAuth2Flow;
import amf.client.model.domain.OAuth2Settings;
import amf.client.model.domain.Parameter;
import amf.client.model.domain.SecurityScheme;

public class AMFSecuritySchemeModel extends APISecuritySchemeModel {

  private final SecurityScheme securityScheme;
  private final OAuth2Flow oAuth2Flow;

  public AMFSecuritySchemeModel(SecurityScheme securityScheme) {
    this(securityScheme, null);
  }

  public AMFSecuritySchemeModel(SecurityScheme securityScheme, OAuth2Flow oAuth2Flow) {
    this.securityScheme = securityScheme;
    this.oAuth2Flow = oAuth2Flow;
    this.name = securityScheme.name().value();
    this.securitySchemeClass = buildSecuritySchemeClass();
    this.customHeaders = buildCustomHeadersModel();
    this.customQueryParams = buildCustomQueryParamsModel();
    this.accessTokenUri = buildAccessTokenUri();
    this.authorizationUri = buildAuthorizationUri();
    this.scopes = buildScopes();
  }

  private String buildAccessTokenUri() {
    if (oAuth2Flow != null) {
      return oAuth2Flow.accessTokenUri().value();
    }
    return null;
  }

  private String buildAuthorizationUri() {
    if (oAuth2Flow != null) {
      return oAuth2Flow.authorizationUri().value();
    }
    return null;
  }

  private List<String> buildScopes() {
    if (oAuth2Flow != null) {
      return oAuth2Flow.scopes().stream().map(x -> x.name().value()).collect(toList());
    }
    return emptyList();
  }

  private Class<? extends ConnectorSecurityScheme> buildSecuritySchemeClass() {
    String schemeType = securityScheme.type().value();

    if (AMFSecuritySchemesNaming.isBasicAuth(schemeType, getHttpSettingsScheme())) {
      return BasicAuthScheme.class;
    } else if (AMFSecuritySchemesNaming.isPassThrough(schemeType)) {
      return PassThroughScheme.class;
    } else if (AMFSecuritySchemesNaming.isApiKey(schemeType)) {
      return PassThroughScheme.class;
    } else if (AMFSecuritySchemesNaming.isOauth2(schemeType)) {
      OAuth2Settings oAuth2Settings = (OAuth2Settings) securityScheme.settings();
      if (AMFOauth2FlowsNaming.isAuthorizationCode(oAuth2Settings, oAuth2Flow)) {
        return OAuth2AuthorizationCodeScheme.class;
      } else if (AMFOauth2FlowsNaming.isClientCredentials(oAuth2Settings, oAuth2Flow)) {
        return OAuth2ClientCredentialsScheme.class;
      }
    } else if (AMFSecuritySchemesNaming.isDigest(schemeType)) {
      return DigestAuthenticationScheme.class;
    } else if (AMFSecuritySchemesNaming.isCustom(schemeType)) {
      return CustomAuthenticationScheme.class;
    }

    return NotSupportedScheme.class;
  }

  private String getHttpSettingsScheme() {
    String httpScheme = null;
    if (securityScheme.settings() instanceof HttpSettings) {
      httpScheme = ((HttpSettings) securityScheme.settings()).scheme().value();
    }
    return httpScheme;
  }

  private List<APIParameterModel> buildCustomQueryParamsModel() {
    List<APIParameterModel> list = new ArrayList<>();
    for (Parameter x : securityScheme.queryParameters()) {
      AMFParameterModel parameterModel = new AMFParameterModel(x, ParameterType.QUERY, false);
      list.add(parameterModel);
    }

    AMFParameterModel queryParam = getApiKeyParameter("query");
    if (queryParam != null) {
      list.add(queryParam);
    }

    return list;
  }

  private List<APIParameterModel> buildCustomHeadersModel() {
    List<APIParameterModel> list = new ArrayList<>();
    for (Parameter x : securityScheme.headers()) {
      AMFParameterModel parameterModel = new AMFParameterModel(x, ParameterType.HEADER, false);
      list.add(parameterModel);
    }

    AMFParameterModel header = getApiKeyParameter("header");
    if (header != null) {
      list.add(header);
    }

    return list;
  }

  private AMFParameterModel getApiKeyParameter(String from) {
    if (securityScheme.settings() instanceof ApiKeySettings) {
      String in = ((ApiKeySettings) securityScheme.settings()).in().value();
      String name = ((ApiKeySettings) securityScheme.settings()).name().value();
      name = isNotBlank(name) ? name : "api-key";

      if (in.equalsIgnoreCase(from)) {
        return new AMFParameterModel(name, ParameterType.SECURITY, new AMFPrimitiveTypeModel(
                                                                                             PrimitiveTypeDefinition.PrimitiveType.STRING));
      }
    }

    return null;
  }
}
