/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.getApiOperationModel;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.getOperationDescriptor;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationRule.Level.ERROR;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.EmptyTypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.PostValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationError;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;

import java.util.List;
import java.util.Objects;

public class PagedOperationsOperationMustDeclareResponseBodyRule extends PostValidationRule {

  public PagedOperationsOperationMustDeclareResponseBodyRule() {
    super("Paged Operations must declare a response body",
          "All paginated operations must declare a an output type.",
          ERROR);
  }

  @Override
  public List<ValidationError> postValidate(ConnectorDescriptor connectorDescriptor,
                                            APIModel apiModel,
                                            ConnectorModel connectorModel) {

    List<APIOperationModel> nullOutputAPIOperations =
        connectorModel.getOperations().stream()
            .filter(connectorOperation -> connectorOperation.getOutputMetadata() == null
                || connectorOperation.getOutputMetadata() instanceof EmptyTypeDefinition)
            .map(connectorOperation -> getApiOperationModel(apiModel, connectorOperation))
            .map(Objects::requireNonNull)
            .collect(toList());

    return nullOutputAPIOperations.stream()
        .filter(nullOutputOperation -> operationIsPaginated(connectorDescriptor, nullOutputOperation))
        .filter(nullOutputOperation -> !typeDefinedInDescriptor(connectorDescriptor, nullOutputOperation))
        .map(nullOutputOperation -> getValidationError(connectorDescriptor, nullOutputOperation))
        .collect(toList());
  }

  private boolean typeDefinedInDescriptor(ConnectorDescriptor connectorDescriptor, APIOperationModel apiOperationModel) {
    final OperationDescriptor operationDescriptor = getOperationDescriptor(connectorDescriptor, apiOperationModel);
    return operationDescriptor != null && isNotBlank(operationDescriptor.getOutputTypeSchema());
  }

  private boolean operationIsPaginated(ConnectorDescriptor connectorDescriptor, APIOperationModel apiOperationModel) {
    final OperationDescriptor operationDescriptor = getOperationDescriptor(connectorDescriptor, apiOperationModel);
    return operationDescriptor != null && isNotBlank(operationDescriptor.getPagination());
  }

  private ValidationError getValidationError(ConnectorDescriptor connectorDescriptor, APIOperationModel apiOperationModel) {
    final OperationDescriptor operationDescriptor = getOperationDescriptor(connectorDescriptor, apiOperationModel);
    return new ValidationError(this, EMPTY, operationDescriptor.getLocation());
  }
}
