/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules;

import static java.util.stream.Collectors.toList;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationRule.Level.ERROR;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.EndPointDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.PreValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationError;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;

import java.util.LinkedList;
import java.util.List;

public class DescriptorPathMustExistInApiRule extends PreValidationRule {

  //R001
  public DescriptorPathMustExistInApiRule() {
    super("Descriptor PATHs must be present in the API spec",
          "Any path declared in the connector descriptor must be present in the API spec.",
          ERROR);
  }

  @Override
  public List<ValidationError> preValidate(ConnectorDescriptor connectorDescriptor, APIModel apiModel) {

    List<String> apiPaths = apiModel.getOperationsModel().stream()
        .map(APIOperationModel::getPath)
        .distinct()
        .collect(toList());

    List<ValidationError> validationErrors = new LinkedList<>();

    for (EndPointDescriptor endPointDescriptor : connectorDescriptor.getEndpoints()) {
      if (apiPaths.stream().noneMatch(x -> x.equalsIgnoreCase(endPointDescriptor.getPath()))) {
        validationErrors.add(getValidationError(endPointDescriptor));
      }
    }

    return validationErrors;
  }

  private ValidationError getValidationError(EndPointDescriptor endPointDescriptor) {
    return new ValidationError(this,
                               "Path: " + endPointDescriptor.getPath() + " is not present in the API spec",
                               endPointDescriptor.getLocation());
  }
}
