/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.getClosestDescriptorElement;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.getOperationDescriptor;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationRule.Level.INFO;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DescriptorElement;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.PreValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationError;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;

import java.util.List;

public class NoDefaultOutputMediaTypeDefinedRule extends PreValidationRule {

  //R015
  public NoDefaultOutputMediaTypeDefinedRule() {
    super("No default outputMediaType was set in the Descriptor and multiple are available",
          EMPTY,
          INFO);
  }

  @Override
  public List<ValidationError> preValidate(ConnectorDescriptor connectorDescriptor, APIModel apiModel) {

    return apiModel.getOperationsModel().stream()
        .filter(x -> x.getOutputMetadataModel().size() > 1)
        .filter(x -> !operationHasDefaultOutputType(connectorDescriptor, x))
        .map(apiOp -> getValidationError(connectorDescriptor, apiOp))
        .collect(toList());
  }

  private boolean operationHasDefaultOutputType(ConnectorDescriptor connectorDescriptor, APIOperationModel apiOperationModel) {
    OperationDescriptor opDesc = getOperationDescriptor(connectorDescriptor, apiOperationModel);

    if (opDesc == null) {
      return false;
    }

    return isNotBlank(opDesc.getOutputMediaType());
  }

  private ValidationError getValidationError(ConnectorDescriptor connectorDescriptor, APIOperationModel apiOperationModel) {
    final String detail =
        "API Operation with PATH: "
            + apiOperationModel.getPath() +
            " and METHOD: "
            + apiOperationModel.getHttpMethod().toUpperCase()
            + " declares multiple output media types but none was selected in the Operation descriptor.";

    final DescriptorElement descriptorElement = getClosestDescriptorElement(connectorDescriptor, apiOperationModel);

    return new ValidationError(this, detail, descriptorElement.getLocation());
  }
}
