/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.HTTPMethod.GET;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.HTTPMethod.OPTIONS;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.HTTPMethod.PATCH;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.HTTPMethod.POST;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.getApiOperationModel;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.getClosestDescriptorElement;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.getOperationDescriptor;
import static java.util.stream.Collectors.toList;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationRule.Level.ERROR;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.EmptyTypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DescriptorElement;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.PostValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationError;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;

import java.util.List;
import java.util.Objects;

public class OperationMustDeclareResponseBodyRule extends PostValidationRule {

  //R003
  public OperationMustDeclareResponseBodyRule() {
    super("Operations must declare a response body",
          "GET, POST, PATCH and OPTIONS operations defined in the API spec must declare a response body and a type for that body. If they don't the connector descriptor must skip this check, define a response body for the operation, or declare it as void.",
          ERROR);
  }

  @Override
  public List<ValidationError> postValidate(ConnectorDescriptor connectorDescriptor,
                                            APIModel apiModel,
                                            ConnectorModel connectorModel) {

    List<APIOperationModel> nullOutputAPIOperations =
        connectorModel.getOperations().stream()
            .filter(connectorOperation -> httpMethodApplies(connectorOperation.getHttpMethod()))
            .filter(connectorOperation -> connectorOperation.getOutputMetadata() == null
                || connectorOperation.getOutputMetadata() instanceof EmptyTypeDefinition)
            .map(connectorOperation -> getApiOperationModel(apiModel, connectorOperation))
            .map(Objects::requireNonNull)
            .collect(toList());

    return nullOutputAPIOperations.stream()
        .filter(nullOutputOperation -> !validationSkipped(connectorModel, nullOutputOperation))
        .filter(nullOutputOperation -> !isVoidOperation(connectorDescriptor, nullOutputOperation))
        .map(nullOutputOperation -> getValidationError(connectorDescriptor, nullOutputOperation))
        .collect(toList());
  }

  private boolean isVoidOperation(ConnectorDescriptor connectorDescriptor, APIOperationModel apiOperationModel) {
    final OperationDescriptor operationDescriptor = getOperationDescriptor(connectorDescriptor, apiOperationModel);

    return operationDescriptor != null && operationDescriptor.getVoidOperation() != null
        && operationDescriptor.getVoidOperation();
  }

  private boolean httpMethodApplies(String httpMethod) {
    return httpMethod.equalsIgnoreCase(GET.name())
        || httpMethod.equalsIgnoreCase(POST.name())
        || httpMethod.equalsIgnoreCase(PATCH.name())
        || httpMethod.equalsIgnoreCase(OPTIONS.name());
  }

  private boolean validationSkipped(ConnectorModel connectorModel, APIOperationModel apiOperationModel) {
    boolean globalSkip = connectorModel.getSkipOutputTypeValidation() != null && connectorModel.getSkipOutputTypeValidation();

    ConnectorOperation connectorOperation = connectorModel.getOperations().stream()
        .filter(x -> x.getHttpMethod().equalsIgnoreCase(apiOperationModel.getHttpMethod()))
        .filter(x -> x.getPath().equalsIgnoreCase(apiOperationModel.getPath()))
        .findFirst().orElse(null);

    if (connectorOperation == null) {
      return false;
    }

    if (connectorOperation.getSkipOutputTypeValidation() == null) {
      return globalSkip;
    } else {
      return connectorOperation.getSkipOutputTypeValidation();
    }
  }

  private ValidationError getValidationError(ConnectorDescriptor connectorDescriptor, APIOperationModel apiOperationModel) {
    final DescriptorElement closestDescriptorElement = getClosestDescriptorElement(connectorDescriptor, apiOperationModel);

    final String detail =
        "Operation with PATH: "
            + apiOperationModel.getPath()
            + " and METHOD: "
            + apiOperationModel.getHttpMethod().toUpperCase()
            + " does not declare a response body.";

    return new ValidationError(this, detail, closestDescriptorElement.getLocation());
  }
}
