/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.HEADER;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.QUERY;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.URI;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationRule.Level.WARN;
import static org.apache.commons.lang3.StringUtils.EMPTY;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TriggerDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TriggerParameterBindingDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TriggerParameterBindingsDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.PreValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationError;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIParameterModel;

import java.util.LinkedList;
import java.util.List;

public class TriggerBindingParameterMustExistInApiRule extends PreValidationRule {

  public TriggerBindingParameterMustExistInApiRule() {
    super("Parameter bindings defined in a trigger must exist in the API spec.", EMPTY, WARN);
  }

  @Override
  public List<ValidationError> preValidate(ConnectorDescriptor connectorDescriptor, APIModel apiModel) {
    final List<ValidationError> results = new LinkedList<>();

    List<TriggerDescriptor> triggerDescriptors = connectorDescriptor.getTriggers();
    if (!triggerDescriptors.isEmpty()) {
      for (TriggerDescriptor triggerDescriptor : connectorDescriptor.getTriggers()) {

        APIOperationModel operationModel = getOperation(apiModel, triggerDescriptor);
        if (operationModel == null) {
          continue;
        }

        TriggerParameterBindingsDescriptor parameterBindings = triggerDescriptor.getParameterBindings();
        analyzeParameters(results, triggerDescriptor, operationModel, parameterBindings.getHeaders(), HEADER);
        analyzeParameters(results, triggerDescriptor, operationModel, parameterBindings.getQueryParameters(), QUERY);
        analyzeParameters(results, triggerDescriptor, operationModel, parameterBindings.getUriParameters(), URI);
      }
    }

    return results;
  }

  private void analyzeParameters(List<ValidationError> results, TriggerDescriptor triggerDescriptor,
                                 APIOperationModel operationModel, List<TriggerParameterBindingDescriptor> bindings,
                                 ParameterType bindingParameterType) {
    for (TriggerParameterBindingDescriptor binding : bindings) {
      if (!hasMatchingParameter(operationModel, binding, bindingParameterType)) {
        results.add(getValidationError(triggerDescriptor, binding, bindingParameterType));
      }
    }
  }

  private boolean hasMatchingParameter(APIOperationModel operationModel, TriggerParameterBindingDescriptor binding,
                                       ParameterType parameterType) {
    final List<APIParameterModel> parameters;
    if (parameterType == HEADER) {
      parameters = operationModel.getHeadersModel();
    } else if (parameterType == QUERY) {
      parameters = operationModel.getQueryParamsModel();
    } else {
      parameters = operationModel.getUriParamsModel();
    }

    return parameters.stream().anyMatch(x -> x.getExternalName().equalsIgnoreCase(binding.getName()));
  }

  private APIOperationModel getOperation(APIModel apiModel, TriggerDescriptor triggerDescriptor) {
    return apiModel.getOperationsModel().stream()
        .filter(x -> x.getPath().equalsIgnoreCase(triggerDescriptor.getPath()))
        .filter(x -> x.getHttpMethod().equalsIgnoreCase(triggerDescriptor.getMethod().getName()))
        .findFirst().orElse(null);
  }

  private ValidationError getValidationError(TriggerDescriptor triggerDescriptor,
                                             TriggerParameterBindingDescriptor binding,
                                             ParameterType parameterType) {
    String detail = "'" +
        parameterType.getName()
        + "' parameter binding '"
        + binding.getName()
        + "' defined in Trigger with PATH: "
        + triggerDescriptor.getPath()
        + " and METHOD: "
        + triggerDescriptor.getMethod().getName().toUpperCase()
        + " declares a binding that does not exist in the API spec.";

    return new ValidationError(this, detail, triggerDescriptor.getLocation());
  }
}
