/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationRule.Level.ERROR;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.EndPointDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ValueResolverDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ValueResolverParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ValueResolverReferenceArgumentDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.PreValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationError;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.LinkedList;
import java.util.List;

public class ValueResolverReferenceArgumentsExistsRule extends PreValidationRule {

  public ValueResolverReferenceArgumentsExistsRule() {
    super("When sending an argument to a value resolver it must exist in the value resolver declaration", EMPTY, ERROR);
  }

  @Override
  public List<ValidationError> preValidate(ConnectorDescriptor connectorDescriptor, APIModel apiModel) {
    final List<ValueResolverDescriptor> valueResolvers = connectorDescriptor.getValueResolvers();

    final List<ValidationError> validationErrors = new LinkedList<>();

    for (EndPointDescriptor endpoint : connectorDescriptor.getEndpoints()) {
      for (OperationDescriptor operation : endpoint.getOperations()) {
        if (operation.getExpects() != null) {
          validationErrors
              .addAll(validateOperationParameters(valueResolvers, endpoint, operation, operation.getExpects().getUriParameter()));
          validationErrors
              .addAll(validateOperationParameters(valueResolvers, endpoint, operation,
                                                  operation.getExpects().getQueryParameter()));
          validationErrors
              .addAll(validateOperationParameters(valueResolvers, endpoint, operation, operation.getExpects().getHeader()));
        }
      }
    }
    return validationErrors;
  }

  private List<ValidationError> validateOperationParameters(List<ValueResolverDescriptor> valueResolvers,
                                                            EndPointDescriptor endpointDescriptor,
                                                            OperationDescriptor operationDescriptor,
                                                            List<ParameterDescriptor> parameterDescriptors) {

    final List<ValidationError> validationErrors = new LinkedList<>();

    for (ParameterDescriptor parameter : parameterDescriptors) {
      if (parameter.getValueResolver() != null && isNotBlank(parameter.getValueResolver().getId())) {
        final ValueResolverDescriptor valueResolver = valueResolvers.stream()
            .filter(x -> x.getName().equals(parameter.getValueResolver().getId()))
            .findFirst().orElse(null);

        if (valueResolver != null) {
          final List<String> valueResolverValidParameterNames = valueResolver.getParameters().stream()
              .map(ValueResolverParameterDescriptor::getName).collect(toList());

          if (parameter.getValueResolver().getArguments() != null && !parameter.getValueResolver().getArguments().isEmpty()) {

            validationErrors.addAll(
                                    parameter.getValueResolver().getArguments().stream()
                                        .filter(x -> valueResolverValidParameterNames.stream()
                                            .noneMatch(y -> y.equals(x.getName())))
                                        .map(x -> getValidationError(endpointDescriptor, operationDescriptor, parameter, x))
                                        .collect(toList()));
          }
        }
      }
    }

    return validationErrors;
  }

  private ValidationError getValidationError(EndPointDescriptor endPointDescriptor,
                                             OperationDescriptor operationDescriptor,
                                             ParameterDescriptor parameterDescriptor,
                                             ValueResolverReferenceArgumentDescriptor argument) {
    final String detail =
        parameterDescriptor.getParamName()
            + " parameter declared for the operation with PATH'"
            + endPointDescriptor.getPath()
            + " and METHOD: "
            + operationDescriptor.getMethod()
            + " in the connector descriptor is declaring an invalid argument for its value resolver: '"
            + argument.getName() + "'";

    return new ValidationError(this, detail, argument.getLocation());
  }
}
