/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationRule.Level.ERROR;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;

import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.EndPointDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ValueResolverDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ValueResolverReferenceArgumentDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.PreValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationError;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.LinkedList;
import java.util.List;
import java.util.regex.Pattern;

import org.apache.commons.lang3.StringUtils;

public class ValueResolverReferenceArgumentsHaveValidFormatRule extends PreValidationRule {

  public ValueResolverReferenceArgumentsHaveValidFormatRule() {
    super("Value Resolver arguments must respect the valid format (i.e. uriParameter.X, queryParameter.X or header.X)", EMPTY,
          ERROR);
  }

  @Override
  public List<ValidationError> preValidate(ConnectorDescriptor connectorDescriptor, APIModel apiModel) {
    final List<ValidationError> validationErrors = new LinkedList<>();

    for (EndPointDescriptor endpoint : connectorDescriptor.getEndpoints()) {
      for (OperationDescriptor operation : endpoint.getOperations()) {
        if (operation.getExpects() != null) {
          validationErrors
              .addAll(validateOperationParameters(endpoint, operation, operation.getExpects().getUriParameter(),
                                                  connectorDescriptor.getValueResolvers()));
          validationErrors
              .addAll(validateOperationParameters(endpoint, operation, operation.getExpects().getQueryParameter(),
                                                  connectorDescriptor.getValueResolvers()));
          validationErrors
              .addAll(validateOperationParameters(endpoint, operation, operation.getExpects().getHeader(),
                                                  connectorDescriptor.getValueResolvers()));
        }
      }
    }
    return validationErrors;
  }

  private List<ValidationError> validateOperationParameters(EndPointDescriptor endpointDescriptor,
                                                            OperationDescriptor operationDescriptor,
                                                            List<ParameterDescriptor> parameterDescriptors,
                                                            List<ValueResolverDescriptor> valueResolverDescriptors) {
    final List<ValidationError> validationErrors = new LinkedList<>();

    for (ParameterDescriptor parameter : parameterDescriptors) {

      if (parameter.getValueResolver() != null
          && parameter.getValueResolver().getArguments() != null
          && !parameter.getValueResolver().getArguments().isEmpty()) {

        if (valueResolverDescriptors.isEmpty()
            || valueResolverDescriptors.stream()
                .noneMatch(x -> x.getName().equalsIgnoreCase(parameter.getValueResolver().getId()))) {
          continue;
        }

        validationErrors.addAll(
                                parameter.getValueResolver().getArguments().stream()
                                    .filter(x -> !validArgumentValueFormat(x.getValue()))
                                    .map(x -> getValidationError(endpointDescriptor, operationDescriptor, parameter, x))
                                    .collect(toList()));

      }
    }

    return validationErrors;
  }

  private static final Pattern PATTERN = Pattern.compile("^(?:uriParameter\\.|queryParameter\\.|header\\.)\\S+$");

  private boolean validArgumentValueFormat(String value) {
    if (StringUtils.isBlank(value)) {
      return false;
    }

    return PATTERN.matcher(value).matches();
  }

  private ValidationError getValidationError(EndPointDescriptor endPointDescriptor,
                                             OperationDescriptor operationDescriptor,
                                             ParameterDescriptor parameterDescriptor,
                                             ValueResolverReferenceArgumentDescriptor argument) {
    final String detail =
        "'" + parameterDescriptor.getParamName() +
            "' parameter declared for the operation with PATH '"
            + endPointDescriptor.getPath()
            + "' and METHOD: '"
            + operationDescriptor.getMethod()
            + "' on the connector descriptor is declaring an invalid argument value for the argument '"
            + argument.getName() + "' : '" + argument.getValue() + "'";

    return new ValidationError(this, detail, argument.getLocation());
  }
}
