/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationRule.Level.ERROR;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.getOperationDescriptor;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.PaginationDeclarationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.PreValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationError;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;

import java.util.LinkedList;
import java.util.List;

public class PagedOperationsParameterMustExistRule extends PreValidationRule {

  public PagedOperationsParameterMustExistRule() {
    super("Paged operations must contain a query parameter corresponding to the pagination parameter declared in the pagination definition",
          EMPTY,
          ERROR);
  }

  @Override
  public List<ValidationError> preValidate(ConnectorDescriptor connectorDescriptor, APIModel apiModel) {
    List<ValidationError> validationErrors = new LinkedList<>();

    for (APIOperationModel apiOperation : apiModel.getOperationsModel()) {
      PaginationDeclarationDescriptor paginationDeclaration = getPaginationDeclaration(connectorDescriptor, apiOperation);
      if (paginationDeclaration != null) {
        String parameterName = paginationDeclaration.getPaginationParameterName();
        if (apiOperation.getQueryParamsModel().stream().noneMatch(x -> x.getExternalName().equalsIgnoreCase(parameterName))) {
          validationErrors
              .add(getValidationError(connectorDescriptor, apiOperation, parameterName, paginationDeclaration.getName()));
        }
      }
    }

    return validationErrors;
  }

  private PaginationDeclarationDescriptor getPaginationDeclaration(ConnectorDescriptor connectorDescriptor,
                                                                   APIOperationModel apiOperationModel) {
    final OperationDescriptor operationDescriptor = getOperationDescriptor(connectorDescriptor, apiOperationModel);

    PaginationDeclarationDescriptor paginationDescriptor = null;
    if (operationDescriptor != null && isNotBlank(operationDescriptor.getPagination())) {
      paginationDescriptor = connectorDescriptor.getPaginations().stream()
          .filter(x -> x.getName().equalsIgnoreCase(operationDescriptor.getPagination()))
          .findFirst().orElse(null);
    }

    return paginationDescriptor;
  }

  private ValidationError getValidationError(ConnectorDescriptor connectorDescriptor, APIOperationModel apiOperationModel,
                                             String parameterName, String paginationName) {
    final OperationDescriptor operationDescriptor = getOperationDescriptor(connectorDescriptor, apiOperationModel);
    String description = "The pagination named '" + paginationName + "' declares a query parameter named '" + parameterName
        + "' that is not present in the API spec for the referencing operation";

    return new ValidationError(this, description, operationDescriptor.getLocation());
  }
}
