/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationRule.Level.ERROR;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.getValueProviderReferenceDescriptors;
import static java.lang.String.format;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isBlank;

import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.common.ArgumentDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.resolvers.ResolverArgumentDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.resolvers.ResolverReferenceDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.valueprovider.ValueProviderDefinitionDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.PreValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationError;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.LinkedList;
import java.util.List;
import java.util.regex.Pattern;

public class ValueProviderReferenceArgumentsHaveValidFormatRule extends PreValidationRule {

  private static final String ERROR_TEMPLATE = "Value '%s' for argument '%s' does not match the expected format.";

  public ValueProviderReferenceArgumentsHaveValidFormatRule() {
    super("Value Provider arguments must respect the valid format (i.e. uriParameter.X, queryParameter.X or header.X)",
          EMPTY,
          ERROR);
  }

  @Override
  public List<ValidationError> preValidate(ConnectorDescriptor connectorDescriptor, APIModel apiModel) {
    final List<ValidationError> validationErrors = new LinkedList<>();

    for (ResolverReferenceDescriptor<ValueProviderDefinitionDescriptor> reference : getValueProviderReferenceDescriptors(connectorDescriptor)) {
      final List<ResolverArgumentDescriptor> arguments = reference.getArguments();

      final List<ValidationError> errors = arguments.stream()
          .filter(x -> !isValidArgumentValueFormat(x.getValue()))
          .map(this::getValidationError)
          .collect(toList());

      validationErrors.addAll(errors);
    }

    return validationErrors;
  }

  private static final Pattern PATTERN = Pattern.compile("^(?:uriParameter\\.|queryParameter\\.|header\\.)\\S+$");

  private boolean isValidArgumentValueFormat(String value) {
    if (isBlank(value)) {
      return false;
    }
    return PATTERN.matcher(value).matches();
  }

  private ValidationError getValidationError(ArgumentDescriptor argument) {
    return new ValidationError(this,
                               format(ERROR_TEMPLATE, argument.getValue(), argument.getName()),
                               argument.getLocation());
  }
}
