/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationRule.Level.ERROR;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.getApiOperationModel;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.getValueProviderReferenceDescriptors;
import static java.lang.String.format;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isBlank;

import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.EndPointDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.common.ArgumentDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.resolvers.ResolverArgumentDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.resolvers.ResolverReferenceDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.valueprovider.ValueProviderDefinitionDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.PreValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationError;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;

import java.util.LinkedList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class ValueProviderReferenceArgumentsOperationParameterExistsRule extends PreValidationRule {

  private static final String ERROR_TEMPLATE = "Argument '%s' uses a parameter that doesnt exist in the API operation: '%s'.";

  public ValueProviderReferenceArgumentsOperationParameterExistsRule() {
    super("All parameter references used as argument for a value provider must exist in the API Operation.",
          EMPTY,
          ERROR);
  }

  @Override
  public List<ValidationError> preValidate(ConnectorDescriptor connectorDescriptor, APIModel apiModel) {
    final List<ValidationError> validationErrors = new LinkedList<>();

    for (EndPointDescriptor endPointDescriptor : connectorDescriptor.getEndpoints()) {
      for (OperationDescriptor operationDescriptor : endPointDescriptor.getOperations()) {
        for (ResolverReferenceDescriptor<ValueProviderDefinitionDescriptor> reference : getValueProviderReferenceDescriptors(operationDescriptor)) {
          final List<ResolverArgumentDescriptor> arguments = reference.getArguments();

          final List<ValidationError> errors = arguments.stream()
              .filter(x -> validArgumentValueFormat(x.getValue()))
              .filter(x -> !argumentParameterExistsInOperation(x.getValue(), getApiOperationModel(apiModel, endPointDescriptor,
                                                                                                  operationDescriptor)))
              .map(this::getValidationError)
              .collect(toList());

          validationErrors.addAll(errors);
        }
      }
    }

    return validationErrors;
  }

  private boolean argumentParameterExistsInOperation(String value, APIOperationModel apiOperationModel) {
    final String parameterType = getParameterTypeReference(value);
    final String parameterName = getParameterReference(value);

    if (parameterType == null || parameterName == null) {
      return false;
    }

    if (parameterType.startsWith("uriParameter")) {
      return apiOperationModel.getUriParamsModel().stream()
          .anyMatch(x -> x.getExternalName().equalsIgnoreCase(parameterName));
    } else if (parameterType.startsWith("queryParameter")) {
      return apiOperationModel.getQueryParamsModel().stream()
          .anyMatch(x -> x.getExternalName().equalsIgnoreCase(parameterName));
    } else if (parameterType.startsWith("header")) {
      return apiOperationModel.getHeadersModel().stream()
          .anyMatch(x -> x.getExternalName().equalsIgnoreCase(parameterName));
    }

    return false;
  }


  private static final Pattern PARAMETER_NAME_PATTERN = Pattern.compile("^(uriParameter\\.|queryParameter\\.|header\\.)(\\S+)$");

  private String getParameterTypeReference(String value) {
    if (isBlank(value)) {
      return null;
    }

    final Matcher matcher = PARAMETER_NAME_PATTERN.matcher(value);
    if (matcher.matches()) {
      return matcher.group(1);
    }
    return null;
  }

  private String getParameterReference(String value) {
    if (isBlank(value)) {
      return null;
    }

    final Matcher matcher = PARAMETER_NAME_PATTERN.matcher(value);
    if (matcher.matches()) {
      return matcher.group(2);
    }
    return null;
  }


  private static final Pattern VALID_FORMAT_PATTERN = Pattern.compile("^(?:uriParameter\\.|queryParameter\\.|header\\.)\\S+$");

  private boolean validArgumentValueFormat(String value) {
    if (isBlank(value)) {
      return false;
    }
    return VALID_FORMAT_PATTERN.matcher(value).matches();
  }


  private ValidationError getValidationError(ArgumentDescriptor argumentDescriptor) {
    return new ValidationError(this,
                               format(ERROR_TEMPLATE, argumentDescriptor.getName(), argumentDescriptor.getValue()),
                               argumentDescriptor.getLocation());
  }
}
