/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.getOperationDescriptor;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationRule.Level.ERROR;

import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.EndPointDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.PreValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationError;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APITypeModel;

import java.util.LinkedList;
import java.util.List;
import java.util.Optional;

public class DescriptorOtputMediaTypeMustBePresentInApiRule extends PreValidationRule {

  // R020
  public DescriptorOtputMediaTypeMustBePresentInApiRule() {
    super("The output media type defined in the descriptor is not present in the API spec",
          "If the API spec defines a set of output media types for an operation the Descriptor must use one of those to define the operation's default output media type.",
          ERROR);
  }

  @Override
  public List<ValidationError> preValidate(ConnectorDescriptor connectorDescriptor, APIModel apiModel) {

    List<ValidationError> allErrors = new LinkedList<>();

    // Check every declared output media type in the operation descriptors against the corresponding API operations
    for (EndPointDescriptor endpointDesc : connectorDescriptor.getEndpoints()) {
      List<ValidationError> errors = endpointDesc.getOperations().stream()
          .filter(opDesc -> isNotBlank(opDesc.getOutputMediaType()))
          .filter(opDesc -> apiDefinesMediaTypes(apiModel, endpointDesc.getPath(), opDesc))
          .filter(opDesc -> !mediaTypePresentInApi(apiModel, endpointDesc.getPath(), opDesc))
          .map(opDesc -> getValidationErrorForOperationDescriptor(endpointDesc.getPath(), opDesc))
          .collect(toList());

      allErrors.addAll(errors);
    }

    // Check the global output media type declared in the connector descriptor against the API operations
    String globalMediaType = connectorDescriptor.getDefaultOutputMediaType();
    if (isNotBlank(globalMediaType)) {
      List<ValidationError> globalErrors = apiModel.getOperationsModel().stream()
          .filter(apiOp -> !apiOp.getOutputMetadataModel().isEmpty())
          .filter(apiOp -> apiOp.getOutputMetadataModel().stream().anyMatch(y -> y.getMediaType() != null))
          .filter(apiOp -> !descriptorDefinesMediaType(connectorDescriptor, apiOp))
          .filter(apiOp -> !descriptorDefinesOperationMediaType(globalMediaType, apiOp))
          .map(apiOp -> getValidationErrorForGlobalDescriptor(apiOp.getPath(), apiOp.getHttpMethod(), globalMediaType,
                                                              connectorDescriptor))
          .collect(toList());

      allErrors.addAll(globalErrors);
    }

    return allErrors;
  }

  protected boolean descriptorDefinesOperationMediaType(String globalMediaType, APIOperationModel apiOperationModel) {
    return apiOperationModel.getOutputMetadataModel().stream()
        .anyMatch(y -> y.getMediaType().toString().equalsIgnoreCase(globalMediaType));
  }

  private boolean descriptorDefinesMediaType(ConnectorDescriptor connectorDescriptor, APIOperationModel apiOperation) {
    final OperationDescriptor operationDescriptor = getOperationDescriptor(connectorDescriptor, apiOperation);
    return operationDescriptor != null && isNotBlank(operationDescriptor.getOutputMediaType());
  }

  private boolean mediaTypePresentInApi(APIModel apiModel, String path, OperationDescriptor operationDesc) {
    String outputMediaType = operationDesc.getOutputMediaType();
    String method = operationDesc.getMethod();

    return apiModel.getOperationsModel().stream()
        .filter(x -> x.getPath().equalsIgnoreCase(path) && x.getHttpMethod().equalsIgnoreCase(method))
        .flatMap(x -> x.getOutputMetadataModel().stream())
        .map(APITypeModel::getMediaType)
        .anyMatch(x -> x.toString().equalsIgnoreCase(outputMediaType));
  }

  private boolean apiDefinesMediaTypes(APIModel apiModel, String path, OperationDescriptor operationDesc) {
    String method = operationDesc.getMethod();

    Optional<APIOperationModel> any = apiModel.getOperationsModel().stream()
        .filter(x -> x.getPath().equalsIgnoreCase(path) && x.getHttpMethod().equalsIgnoreCase(method))
        .filter(x -> !x.getOutputMetadataModel().isEmpty())
        .findAny();

    return any.isPresent();
  }

  private ValidationError getValidationErrorForOperationDescriptor(String path, OperationDescriptor operationDescriptor) {
    String location =
        "API Operation with PATH: "
            + path
            + " and METHOD: "
            + operationDescriptor.getMethod().toUpperCase()
            + " does not declare '"
            + operationDescriptor.getOutputMediaType()
            + "' output media type that is used in the Operation Descriptor";

    return new ValidationError(this, location, operationDescriptor.getLocation());
  }

  private ValidationError getValidationErrorForGlobalDescriptor(String path, String method, String mediaType,
                                                                ConnectorDescriptor connectorDescriptor) {
    String location =
        "API Operation with PATH: "
            + path
            + " and METHOD: "
            + method
            + " does not declare '"
            + mediaType
            + "' output media type that is used as global output media type in the Connector Descriptor";

    return new ValidationError(this, location, connectorDescriptor.getLocation());
  }
}
