/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.HTTPMethod.GET;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationRule.Level.WARN;
import static org.apache.commons.lang3.StringUtils.EMPTY;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.HTTPMethod;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.EndPointDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.HttpMethodDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TriggerDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.resolvers.ResolverExpressionDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.resolvers.ResolverReferenceDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.sampledata.SampleDataDefinitionDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.sampledata.SampleDataSameOperationDefinitionDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.PreValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationError;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.ArrayList;
import java.util.List;

public class SampleDataMarkerMustBeGETRule extends PreValidationRule {

  public SampleDataMarkerMustBeGETRule() {
    super("Sample Data Same Operation cannot be used in POST/PUT/DELETE/PATCH methods, ignoring this one.",
          "Sample Data Same Operation is only available for GET method, others will be ignored.",
          WARN);
  }

  @Override
  public List<ValidationError> preValidate(ConnectorDescriptor connectorDescriptor, APIModel apiModel) {

    List<ValidationError> results = new ArrayList<>();

    for (EndPointDescriptor endpoint : connectorDescriptor.getEndpoints()) {
      for (OperationDescriptor operation : endpoint.getOperations()) {
        String method = operation.getMethod();
        if (sampleDataMarkerWithoutGET(operation.getSampleDataExpressionDescriptor(), method)) {
          ValidationError vr = new ValidationError(this, EMPTY, operation.getLocation());
          results.add(vr);
        }
      }
    }

    for (TriggerDescriptor trigger : connectorDescriptor.getTriggers()) {
      HttpMethodDescriptor method = trigger.getMethod();
      if (sampleDataMarkerWithoutGET(trigger.getSampleDataExpressionDescriptor(), method.getName())) {
        ValidationError vr = new ValidationError(this, EMPTY, trigger.getLocation());
        results.add(vr);
      }
    }
    return results;
  }

  private boolean sampleDataMarkerWithoutGET(ResolverExpressionDescriptor<SampleDataDefinitionDescriptor> resolverExpression,
                                             String method) {
    if (resolverExpression instanceof ResolverReferenceDescriptor) {
      resolverExpression =
          ((ResolverReferenceDescriptor<SampleDataDefinitionDescriptor>) resolverExpression).getDeclaration().getDefinition();
    }

    if (resolverExpression instanceof SampleDataSameOperationDefinitionDescriptor) {
      return HTTPMethod.fromString(method.toUpperCase()) != GET;
    }

    return false;
  }
}
