/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationRule.Level.ERROR;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.OperationIdentifierExpressionHandler;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.PostValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationError;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

public class OperationIdentifierScriptMustNotRepeatRule extends PostValidationRule {

  public OperationIdentifierScriptMustNotRepeatRule() {
    super("Operation identifier expression must not repeat values among operations",
          String.format("The 'operationIdentifier' must be revisited (available bindings are '%s')",
                        Arrays.toString(OperationIdentifierExpressionHandler.IMPLICIT_INPUTS)),
          ERROR);
  }

  @Override
  public List<ValidationError> postValidate(ConnectorDescriptor connectorDescriptor, APIModel apiModel,
                                            ConnectorModel connectorModel) {
    Map<String, List<ConnectorOperation>> operationIdentifiers = new HashMap<>();
    connectorModel.getOperations().forEach(op -> {
      String operationIdentifier = op.getOperationIdentifier();
      if (!operationIdentifiers.containsKey(operationIdentifier)) {
        operationIdentifiers.put(operationIdentifier, new ArrayList<>());
      }
      operationIdentifiers.get(operationIdentifier).add(op);
    });

    List<ValidationError> result = new ArrayList<>();
    operationIdentifiers.entrySet().forEach(entry -> {
      if (entry.getValue().size() > 1) {
        result.add(getValidationError(connectorDescriptor, entry.getKey(), entry.getValue()));
      }
    });

    return result;
  }

  private ValidationError getValidationError(ConnectorDescriptor connectorDescriptor, String operationIdentifier,
                                             List<ConnectorOperation> operations) {
    String detail = String
        .format("The generated 'operationIdentifier' [%s] is repeated for the following operations [%s]. Please, revisit the 'operationIdentifier' script",
                operationIdentifier,
                operations.stream().map(op -> op.getHttpMethod() + ":" + op.getPath()).collect(Collectors.joining(", ")));
    return new ValidationError(this, detail, connectorDescriptor.getOperationIdentifier().getLocation());
  }

}
