/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.NamingUtil.isFriendlyName;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationRule.Level.WARN;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.getClosestDescriptorElement;
import static java.util.stream.Collectors.toList;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DescriptorElement;
import com.mulesoft.connectivity.rest.sdk.internal.validation.PostValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationError;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.List;

public class OperationNameMustBeFriendlyRule extends PostValidationRule {

  // R012
  public OperationNameMustBeFriendlyRule() {
    super("Operation display name seems not to be a friendly name",
          "The displayName should be a human readable name separated with spaces.",
          WARN);
  }

  @Override
  public List<ValidationError> postValidate(ConnectorDescriptor connectorDescriptor,
                                            APIModel apiModel,
                                            ConnectorModel connectorModel) {

    return connectorModel.getOperations().stream()
        .filter(x -> !isFriendlyName(x.getDisplayName()))
        .map(x -> getValidationError(connectorDescriptor, x))
        .collect(toList());
  }

  private ValidationError getValidationError(ConnectorDescriptor connectorDescriptor, ConnectorOperation connectorOperation) {
    final DescriptorElement descriptorElement = getClosestDescriptorElement(connectorDescriptor, connectorOperation);

    final String detail =
        "Operation with PATH: "
            + connectorOperation.getPath() +
            " and METHOD: "
            + connectorOperation.getHttpMethod().toUpperCase() +
            " display name is '"
            + connectorOperation.getDisplayName()
            + "' and it is not friendly";

    return new ValidationError(this, detail, descriptorElement.getLocation());
  }
}
