/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules;

import static com.mulesoft.connectivity.rest.commons.internal.RestConstants.ATTRIBUTES_VAR;
import static com.mulesoft.connectivity.rest.commons.internal.RestConstants.LINK;
import static com.mulesoft.connectivity.rest.commons.internal.RestConstants.PAYLOAD_VAR;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationRule.Level.ERROR;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.getOperationDescriptor;
import static java.lang.String.format;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import org.mule.weave.v2.parser.phase.CompilationException;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.ExpressionHandlerUtils;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ExpressionDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.PaginationDeclarationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.PreValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationError;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;

import java.util.LinkedList;
import java.util.List;

public class PagedOperationsExpressionMustCompileRule extends PreValidationRule {

  private static final String[] IMPLICIT_INPUTS = {PAYLOAD_VAR, ATTRIBUTES_VAR};
  private static final String[] IMPLICIT_INPUTS_HYPERMEDIA = {PAYLOAD_VAR, ATTRIBUTES_VAR, LINK};

  public PagedOperationsExpressionMustCompileRule() {
    super("Expression is invalid",
          EMPTY,
          ERROR);
  }

  @Override
  public List<ValidationError> preValidate(ConnectorDescriptor connectorDescriptor, APIModel apiModel) {
    List<ValidationError> validationErrors = new LinkedList<>();

    for (APIOperationModel apiOperation : apiModel.getOperationsModel()) {
      PaginationDeclarationDescriptor paginationDeclaration = getPaginationDeclaration(connectorDescriptor, apiOperation);
      if (paginationDeclaration != null) {
        if (paginationDeclaration.getNextTokenExpression() != null) {
          validateExpression(connectorDescriptor, apiOperation, paginationDeclaration,
                             paginationDeclaration.getNextTokenExpression(), IMPLICIT_INPUTS, "nextToken",
                             validationErrors);
        }
        if (paginationDeclaration.getPagingResponseExpression() != null) {
          validateExpression(connectorDescriptor, apiOperation, paginationDeclaration,
                             paginationDeclaration.getPagingResponseExpression(),
                             IMPLICIT_INPUTS, "pagingResponse",
                             validationErrors);
        }
        if (paginationDeclaration.getPageCountExpression() != null) {
          validateExpression(connectorDescriptor, apiOperation, paginationDeclaration,
                             paginationDeclaration.getPageCountExpression(), IMPLICIT_INPUTS, "pageCount",
                             validationErrors);
        }
        if (paginationDeclaration.getNextUrlExpression() != null) {
          validateExpression(connectorDescriptor, apiOperation, paginationDeclaration,
                             paginationDeclaration.getNextUrlExpression(),
                             IMPLICIT_INPUTS_HYPERMEDIA, "nextUrl",
                             validationErrors);
        }
      }
    }

    return validationErrors;
  }

  private void validateExpression(ConnectorDescriptor connectorDescriptor, APIOperationModel apiOperationModel,
                                  PaginationDeclarationDescriptor paginationDeclarationDescriptor,
                                  ExpressionDescriptor expression, String[] implicitInputs, String expressionName,
                                  List<ValidationError> validationErrors) {
    try {
      ExpressionHandlerUtils.compileDataWeaveScript(expression.getExpression(), implicitInputs);
    } catch (CompilationException e) {
      validationErrors.add(getValidationError(connectorDescriptor, apiOperationModel, paginationDeclarationDescriptor,
                                              expressionName, expression.getExpression(), e));
    }
  }

  private PaginationDeclarationDescriptor getPaginationDeclaration(ConnectorDescriptor connectorDescriptor,
                                                                   APIOperationModel apiOperationModel) {
    final OperationDescriptor operationDescriptor = getOperationDescriptor(connectorDescriptor, apiOperationModel);

    PaginationDeclarationDescriptor paginationDescriptor = null;
    if (operationDescriptor != null && isNotBlank(operationDescriptor.getPagination())) {
      paginationDescriptor = connectorDescriptor.getPaginations().stream()
          .filter(x -> x.getName().equalsIgnoreCase(operationDescriptor.getPagination()))
          .findFirst().orElse(null);
    }

    return paginationDescriptor;
  }

  private ValidationError getValidationError(ConnectorDescriptor connectorDescriptor, APIOperationModel apiOperationModel,
                                             PaginationDeclarationDescriptor paginationDeclarationDescriptor,
                                             String expressionName, String expression, CompilationException e) {
    String detail = format("'%s' expression '%s' in %s is not valid, error: \n%s", expressionName, expression,
                           paginationDeclarationDescriptor.getName(), e.getMessage());
    final OperationDescriptor operationDescriptor = getOperationDescriptor(connectorDescriptor, apiOperationModel);
    return new ValidationError(this, detail, operationDescriptor.getLocation());
  }
}
