/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.getApiOperation;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationRule.Level.ERROR;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TriggerDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.PreValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationError;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.List;

public class TriggerPathAndMethodMustExistInApiRule extends PreValidationRule {

  public TriggerPathAndMethodMustExistInApiRule() {
    super("Trigger PATH and METHOD must exist in the API specification.", EMPTY, ERROR);
  }

  @Override
  public List<ValidationError> preValidate(ConnectorDescriptor connectorDescriptor, APIModel apiModel) {

    List<TriggerDescriptor> triggerDescriptors = connectorDescriptor.getTriggers();
    if (triggerDescriptors != null && !triggerDescriptors.isEmpty()) {
      return triggerDescriptors.stream()
          .filter(x -> !hasMatchingOperation(apiModel, x))
          .map(this::getValidationError)
          .collect(toList());
    }

    return emptyList();
  }

  private static boolean hasMatchingOperation(APIModel apiModel, TriggerDescriptor triggerDescriptor) {
    return getApiOperation(apiModel, triggerDescriptor) != null;
  }

  private ValidationError getValidationError(TriggerDescriptor triggerDescriptor) {
    String detail =
        "Trigger with PATH: "
            + triggerDescriptor.getPath()
            + " and METHOD: "
            + triggerDescriptor.getMethod().getName().toUpperCase()
            + " does not exist in the API spec";

    return new ValidationError(this, detail, triggerDescriptor.getLocation());
  }
}
