/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.api;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorPackage.buildBasePackage;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.NamingUtil.isFriendlyName;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.NamingUtil.makeNameFriendly;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.util.XmlUtils.getXmlName;
import static java.lang.String.format;
import static java.nio.charset.StandardCharsets.UTF_8;
import static java.util.Collections.EMPTY_LIST;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.io.FileUtils.writeStringToFile;
import static org.apache.commons.lang.StringUtils.EMPTY;
import static org.apache.commons.lang.StringUtils.isEmpty;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.apache.logging.log4j.LogManager.getLogger;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorCategory;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.NamingUtil;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.APISpecDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.APIUrlDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.BaseUriDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DescriptorElementLocation;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.EndPointDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.MavenGavDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.PaginationDeclarationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.RequestDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.SecuritySchemeBaseDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.writer.DescriptorWriter;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.exception.ModelGenerationException;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIParameterModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APIType;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;

import com.mulesoft.connectivity.rest.sdk.internal.webapi.parser.amf.AMFAPIModelFactory;
import org.apache.logging.log4j.Logger;

public class DescriptorScaffolder {

  private static final Logger LOGGER = getLogger(DescriptorScaffolder.class);

  private final static String DEFAULT_MVN_GROUP_ID = "com.mulesoft.connectors";
  private final static String DEFAULT_MVN_VERSION = "1.0.0-SNAPSHOT";

  private final static String DESCRIPTOR_FILE_NAME_DEFAULT = "descriptor.yaml";
  private final static String DESCRIPTOR_FILE_NAME_TEMPLATE = "descriptor-%s.yaml";


  private DescriptorScaffolder() {}

  public static void scaffoldDescriptor(File spec, Path outputDir,
                                        DescriptorMode descriptorMode, boolean skipValidation)
      throws ModelGenerationException, IOException {
    ConnectorDescriptor connectorDescriptor = scaffoldDescriptorModel(spec, descriptorMode, skipValidation);
    writeToYaml(connectorDescriptor, outputDir);
  }

  public static ConnectorDescriptor scaffoldDescriptorModel(File spec, DescriptorMode descriptorMode,
                                                            boolean skipValidation)
      throws ModelGenerationException {
    APIModel apiModel = new AMFAPIModelFactory().build(spec.toPath(), skipValidation);
    return generateConnectorDescriptor(spec.toPath(), apiModel, descriptorMode);
  }

  public static void writeToYaml(ConnectorDescriptor connectorDescriptor, Path outputDir)
      throws IOException {
    String yaml = DescriptorWriter.generate(connectorDescriptor);

    writeStringToFile(getOutputFile(outputDir), yaml, UTF_8);
  }

  private static File getOutputFile(Path outputDir) {
    File outputFileCandidate = outputDir.resolve(DESCRIPTOR_FILE_NAME_DEFAULT).toFile();
    if (outputFileCandidate.exists()) {
      LOGGER.warn(format("File '%s' already exists.", outputFileCandidate.toString()));
    }

    int i = 1;
    while (outputFileCandidate.exists()) {
      outputFileCandidate = outputDir.resolve(format(DESCRIPTOR_FILE_NAME_TEMPLATE, i++)).toFile();
    }

    LOGGER.info(format("Writing descriptor to '%s'.", outputFileCandidate.toString()));
    return outputFileCandidate;
  }

  private static ConnectorDescriptor generateConnectorDescriptor(Path specPath, APIModel apiModel,
                                                                 DescriptorMode descriptorMode) {
    return ConnectorDescriptor.builder()
        .apiSpec(descriptorMode.isGenerateApiSpec() ? generateAPISpec(specPath) : null)
        .connectorName(descriptorMode.isGenerateApiName() ? generateApiName(apiModel) : null)
        .connectorDescription(descriptorMode.isGenerateDescription() ? apiModel.getDescription() : null)
        .connectorGav(descriptorMode.isGenerateConnectorGav() ? generateConnectorGav(apiModel) : null)
        .baseUri(descriptorMode.isGenerateBaseUri() ? generateBaseUri(apiModel) : null)
        .endpoints(descriptorMode.isGenerateEndpoints() ? generateEndpoints(apiModel) : EMPTY_LIST)
        .paginations(descriptorMode.isGeneratePaginations() ? generatePaginations() : EMPTY_LIST)
        .security(descriptorMode.isGenerateSecurity() ? generateSecurity() : EMPTY_LIST)
        .connectorCategory(ConnectorCategory.SELECT.toString())
        .baseJavaPackage(descriptorMode.isGenerateDefaultPackage() ? generateDefaultPackage(apiModel) : null)
        .extensionXml(descriptorMode.isGenerateExtensionXml() ? getXmlName(generateApiName(apiModel)) : null)
        .operationAdapterDescriptors(emptyList())
        .location(DescriptorElementLocation.EMPTY)
        .build();
  }

  private static String generateDefaultPackage(APIModel apiModel) {
    return buildBasePackage(generateApiName(apiModel));
  }

  private static String generateApiName(APIModel apiModel) {
    String notApiName = apiModel.getApiName()
        .replaceAll("[\\s]+(?i)api[\\s]+", EMPTY)
        .replaceAll("^(?i)api[\\s]+", EMPTY)
        .replaceAll("[\\s]+(?i)api$", EMPTY);

    return makeNameFriendly(notApiName);
  }

  private static List<SecuritySchemeBaseDescriptor> generateSecurity() {
    return emptyList();
  }

  private static List<EndPointDescriptor> generateEndpoints(APIModel apiModel) {
    return apiModel.getOperationsModel().stream()
        .map(APIOperationModel::getPath)
        .distinct()
        .map(x -> generateEndpoint(x, apiModel))
        .collect(toList());
  }

  private static EndPointDescriptor generateEndpoint(String path, APIModel apiModel) {
    return new EndPointDescriptor(path, generateOperations(path, apiModel), false, null,
                                  DescriptorElementLocation.EMPTY);
  }

  private static List<OperationDescriptor> generateOperations(String path, APIModel apiModel) {
    return apiModel.getOperationsModel().stream()
        .filter(x -> x.getPath().equalsIgnoreCase(path))
        .map(DescriptorScaffolder::generateOperation)
        .collect(toList());
  }

  private static OperationDescriptor generateOperation(APIOperationModel operationModel) {
    return OperationDescriptor.builder()
        .method(operationModel.getHttpMethod().toLowerCase())
        .displayName(generateOperationDisplayName(operationModel))
        .description(operationModel.getDescription())
        .inputMediaType(generateDefaultInputMediaType(operationModel))
        .outputMediaType(generateDefaultOutputMediaType(operationModel))
        .expects(generateRequest(operationModel))
        .ignored(false)
        .skipOutputTypeValidation(isBodylessResponse(operationModel))
        .voidOperation(isBodylessResponse(operationModel))
        .location(DescriptorElementLocation.EMPTY)
        .build();
  }

  private static String generateOperationDisplayName(APIOperationModel operationModel) {
    if (isEmpty(operationModel.getName())) {
      return null;
    }

    return isFriendlyName(operationModel.getName())
        ? operationModel.getName()
        : makeNameFriendly(operationModel.getName());
  }

  private static Boolean isBodylessResponse(APIOperationModel operationModel) {
    boolean methodRequiresBody =
        operationModel.getHttpMethod().equalsIgnoreCase("GET")
            || operationModel.getHttpMethod().equalsIgnoreCase("POST")
            || operationModel.getHttpMethod().equalsIgnoreCase("PATCH")
            || operationModel.getHttpMethod().equalsIgnoreCase("OPTIONS");

    if (methodRequiresBody && operationModel.getOutputMetadataModel().isEmpty())
      return false;

    boolean emptyBody = operationModel.getOutputMetadataModel().stream()
        .allMatch(x -> x.getApiType() == null || x.getApiType().equals(APIType.EMPTY));

    if (methodRequiresBody && emptyBody) {
      return false;
    }

    return null;
  }

  private static String generateDefaultInputMediaType(APIOperationModel operationModel) {
    if (!operationModel.getInputMetadataModel().isEmpty()
        && operationModel.getInputMetadataModel().get(0).getMediaType() != null) {
      return operationModel.getInputMetadataModel().get(0).getMediaType().toString();
    }
    return null;
  }

  private static String generateDefaultOutputMediaType(APIOperationModel operationModel) {
    if (!operationModel.getOutputMetadataModel().isEmpty()
        && operationModel.getOutputMetadataModel().get(0).getMediaType() != null) {
      return operationModel.getOutputMetadataModel().get(0).getMediaType().toString();
    }
    return null;
  }

  private static RequestDescriptor generateRequest(APIOperationModel operationModel) {
    return new RequestDescriptor(generateParameters(operationModel.getHeadersModel()),
                                 generateParameters(operationModel.getQueryParamsModel()),
                                 generateParameters(operationModel.getUriParamsModel()),
                                 new ArrayList<>(),
                                 null,
                                 DescriptorElementLocation.EMPTY);
  }

  private static List<ParameterDescriptor> generateParameters(List<APIParameterModel> parametersModel) {
    return parametersModel.stream().map(DescriptorScaffolder::generateParameter).collect(toList());
  }

  private static ParameterDescriptor generateParameter(APIParameterModel parameterModel) {
    return new ParameterDescriptor(generateParameterDisplayName(parameterModel),
                                   parameterModel.getExternalName(),
                                   parameterModel.getDescription(),
                                   false,
                                   null,
                                   null,
                                   DescriptorElementLocation.EMPTY,
                                   parameterModel.getSummary());
  }

  private static String generateParameterDisplayName(APIParameterModel parameterModel) {
    String displayName;

    if (isNotBlank(parameterModel.getDisplayName())) {
      displayName = parameterModel.getDisplayName();
    } else {
      displayName = parameterModel.getExternalName();
      if (!isFriendlyName(displayName)) {
        displayName = NamingUtil.makeNameFriendly(displayName);
      }
    }
    return displayName;
  }

  private static List<PaginationDeclarationDescriptor> generatePaginations() {
    return emptyList();
  }

  private static BaseUriDescriptor generateBaseUri(APIModel apiModel) {
    return new BaseUriDescriptor(apiModel.getBaseUri(), "parameter", emptyList(), DescriptorElementLocation.EMPTY);
  }

  private static MavenGavDescriptor generateConnectorGav(APIModel apiModel) {
    return new MavenGavDescriptor(DEFAULT_MVN_GROUP_ID,
                                  getXmlName(apiModel.getApiName()),
                                  DEFAULT_MVN_VERSION,
                                  DescriptorElementLocation.EMPTY);
  }

  private static APISpecDescriptor generateAPISpec(Path specPath) {
    return new APIUrlDescriptor(specPath.toString(), null);
  }
}
