/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.INFO;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.WARN;
import static java.lang.System.lineSeparator;
import static java.util.stream.Collectors.toList;
import static org.slf4j.LoggerFactory.getLogger;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.ConnectorModelBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ApiDescriptorsValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ApiValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel.ConnectorMustHaveOneOperationOrTriggerRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.OperationAdapterMaxOneCustomFieldRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.OperationAdapterMaxOneMetadataKeyRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.OperationCustomFieldBindingMustBeMetadataKeyRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.OperationCustomFieldBindingMustCompileRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.OperationCustomFieldTransformationScriptMustCompileRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.TriggerDataWeaveExpressionsMustCompileRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.OverrideParameterDescriptorMustBePresentInApiRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.CompleteConnectorModelBuilderValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ConnectorModelCreateConnectorMojoValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ConnectorModelValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorConnectorModelBuilderValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.FilesValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.StepConnectorModelBuilderValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.api.OauthAccessTokenMustExistRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.api.OauthAuthorizationUrlMustExistRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel.AtLeastOneSupportedSecuritySchemeRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel.BodyIdentifierScriptMustNotRepeatRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel.ConnectorMustHaveCompleteGavRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel.ConnectorMustHaveRootBaseUriRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel.NotSupportedSecuritySchemeRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel.OperationIdentifierScriptMustNotRepeatRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel.OperationInternalNameMustNotRepeatRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel.OperationNameMustBeFriendlyRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel.OperationSampleDataAutogeneratedMustBeAccepted;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel.ParameterInternalNameMustNotRepeatRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel.ParametersIdentifiersMustNotRepeatRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel.TriggerDataWeaveExpressionsShouldReturnExpectedMetadataTypeRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel.builder.complete.NoDefaultInputMediaTypeDefinedRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel.builder.complete.NoDefaultOutputMediaTypeDefinedRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel.builder.complete.NoInputMediaTypeDefinedRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel.builder.complete.NoOutputMediaTypeDefinedRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel.builder.complete.OperationMustDeclareResponseBodyRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel.builder.complete.PagedOperationsOperationMustDeclareResponseBodyRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel.builder.step.TriggerMandatoryFieldsRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.BaseJavaPackageMustBeValidRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.connectormodelbuilder.BodyIdentifierExpressionNotOverrideableRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.BodyIdentifierScriptMustCompileRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.ConnectorNameMustBeFriendlyRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.DescriptorAdditionalParameterMandatoryFieldsRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.DescriptorAdditionalParameterMustNotBePresentInApiRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.DescriptorInputMediaTypeMustBePresentInApiRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.DescriptorOperationMustExistInApiRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.DescriptorOtputMediaTypeMustBePresentInApiRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.DescriptorParameterMustBePresentInApiRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.DescriptorParamsMustBeUniquelyTypedRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.DescriptorParamsShouldBeTypedRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.DescriptorPathMustExistInApiRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.DescriptorRedundantOutputDefinitionRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.DescriptorReservedWordsRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.DynamicOperationParametersMustNotBeTypedRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.DynamicTriggerParametersMustNotBeTypedRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.ExtensionXmlMustBeValidRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.GavArtifactIdMustBeValidRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.FileSchemeExistRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.GavGroupIdMustBeValidRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.GavVersionMustBeValidRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.IdentifiersScriptsMustNotBeEmptyRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.OperationAdapterBindingParameterWithValueMustNotBeIgnoredRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.OperationAdapterBodyBindingShouldDefineBoundaryOnMultipartFormDataRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.OperationAdapterDifferentOutputMediaTypeRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.OperationAdapterOutputSchemaWithoutMediaTypeRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.OperationAdapterRequiredParametersMustNotBeIgnoredRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.OperationDisplayNameScriptMustCompileRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.connectormodelbuilder.OperationIdentifierExpressionNotOverrideableRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.OperationIdentifierScriptMustCompileRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.PagedOperationsExpressionMustCompileRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.PagedOperationsParameterMustExistRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.PagedParametersMustNotBeIgnoredRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.connectormodelbuilder.ParameterIdentifierExpressionNotOverrideableRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.ParameterIdentifierScriptMustCompileRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.ParametersUsedInOperationAdapterBindingsShouldExistRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.RequiredParametersMustNotBeIgnoredRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.SampleDataArgumentsOptionalityValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.SampleDataInlineBindingParametersExistsRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.SampleDataReferenceArgumentsExistsRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.SampleDataReferenceBindingParametersExistsRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.SampleDataReferenceParametersMustReferencedRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.SecurityNotSupportedInApiMustHaveAnAssociatedCustomTypeInDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.SecuritySchemeMustBeSupportedByTheApiRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.SecurityTypeMustBeTheSameRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.TestConnectionExpressionMustCompileRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.TriggerBindingParameterMustExistInApiRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.TriggerBaseEndpointMustBeDefinedRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.TriggerCircularDependenciesRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.TriggerParameterSetAsMetadataKeyIdCannotHaveValueProviderRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.TriggerPathAndMethodMustExistInApiOrOperationAdapterRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.TriggerUriParamsMustBeCompleteRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.connectormodelbuilder.TriggerBaseFromFqnNotAllowedRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.ValueProviderBodyLevelInlineBindingParametersExistsRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.ValueProviderBodyLevelReferenceArgumentsHaveValidFormatRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.ValueProviderBodyLevelReferenceBindingParametersExistsRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.ValueProviderInlineBindingParametersExistsRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.ValueProviderMustHaveRequestOrScriptDefinitionRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.ValueProviderReferenceArgumentsExistsRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.ValueProviderReferenceArgumentsHaveValidFormatRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.ValueProviderReferenceArgumentsOperationParametersExistsRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.ValueProviderReferenceBindingParametersExistsRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.ValueProviderReferenceParametersMustReferencedRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.XmlAndJSONSchemasReferencedByDescriptorMustBeWellFormedRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.MockedSecuritySchemeMustBeUniqueRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.DescriptorOperationMultipartResponseNotSupportedRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.connectormodelbuilder.BaseOperationMustExistRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.connectormodelbuilder.BaseTriggerMustExistRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.connectormodelbuilder.OperationAdapterRequestBindingParametersMustExistsRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.files.FileInOverrideMustExistInGeneratedCodeRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.files.OnlyRefinementsGoInRefinementFolderRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.files.RefinedClassesMustBeMarkedInDescriptorFolderRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.files.RefinementsDontGoInOverrideFolderRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;


import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;

public class ValidationEngine {

  private static final Logger LOGGER = getLogger(ValidationEngine.class);

  private final List<ValidationResult> validationResults = new ArrayList<>();
  private final List<ValidationRule> rules;

  public ValidationEngine() {
    this.rules = ALL_RULES;
  }

  public ValidationEngine(List<ValidationRule> rules) {
    this.rules = rules;
  }

  public void validateApi(APIModel apiModel) {
    List<ValidationResult> apiResults = rules.stream()
        .filter(x -> x instanceof ApiValidationRule)
        .flatMap(x -> ((ApiValidationRule) x).validate(apiModel).stream())
        .collect(toList());

    validationResults.addAll(apiResults);
  }

  public void validateDescriptor(APIModel apiModel, ConnectorDescriptor connectorDescriptor,
                                 ConnectorModelBuilder connectorModelBuilder) {
    List<ValidationResult> descriptorResults = rules.stream()
        .filter(x -> x instanceof DescriptorValidationRule)
        .flatMap(x -> ((DescriptorValidationRule) x).validate(apiModel, connectorDescriptor).stream())
        .collect(toList());

    rules.stream()
        .filter(x -> x instanceof DescriptorConnectorModelBuilderValidationRule)
        .flatMap(x -> ((DescriptorConnectorModelBuilderValidationRule) x)
            .validate(apiModel, connectorDescriptor, connectorModelBuilder).stream())
        .forEach(descriptorResults::add);
    validationResults.addAll(descriptorResults);
  }

  public void validateApiDescriptors(APIModel apiModel, List<ConnectorDescriptor> connectorDescriptors) {
    List<ValidationResult> descriptorResults = rules.stream()
        .filter(x -> x instanceof ApiDescriptorsValidationRule)
        .flatMap(x -> ((ApiDescriptorsValidationRule) x).validate(apiModel, connectorDescriptors).stream())
        .collect(toList());

    validationResults.addAll(descriptorResults);
  }


  public void validateConnectorModelBuilderStep(ConnectorModelBuilder connectorModelBuilder) {
    List<ValidationResult> connectorResults = rules.stream()
        .filter(x -> x instanceof StepConnectorModelBuilderValidationRule)
        .flatMap(x -> ((StepConnectorModelBuilderValidationRule) x).validate(connectorModelBuilder).stream())
        .collect(toList());

    validationResults.addAll(connectorResults);
  }

  public void validateConnectorModelBuilder(ConnectorModelBuilder connectorModelBuilder) {
    List<ValidationResult> connectorResults = rules.stream()
        .filter(x -> x instanceof CompleteConnectorModelBuilderValidationRule)
        .flatMap(x -> ((CompleteConnectorModelBuilderValidationRule) x).validate(connectorModelBuilder).stream())
        .collect(toList());

    validationResults.addAll(connectorResults);
  }

  public void validateConnectorModel(APIModel apiModel, ConnectorModel connectorModel) {
    List<ValidationResult> connectorResults = rules.stream()
        .filter(x -> x instanceof ConnectorModelValidationRule)
        .flatMap(x -> ((ConnectorModelValidationRule) x).validate(apiModel, connectorModel).stream())
        .collect(toList());

    validationResults.addAll(connectorResults);
  }

  public void validateConnectorModelCreateConnectorMojo(ConnectorModel connectorModel) {
    List<ValidationResult> connectorResults = rules.stream()
        .filter(x -> x instanceof ConnectorModelCreateConnectorMojoValidationRule)
        .flatMap(x -> ((ConnectorModelCreateConnectorMojoValidationRule) x).validate(connectorModel).stream())
        .collect(toList());

    validationResults.addAll(connectorResults);
  }

  public void validateFiles(Path outputDir, RestSdkRunConfiguration runConfiguration) {
    List<ValidationResult> connectorResults = rules.stream()
        .filter(x -> x instanceof FilesValidationRule)
        .flatMap(x -> ((FilesValidationRule) x).validate(outputDir, runConfiguration).stream())
        .collect(toList());

    validationResults.addAll(connectorResults);
  }

  public boolean isSuccess() {
    return validationResults.stream().noneMatch(x -> x.getLevel().equals(ERROR));
  }

  public void logResults() {
    validationResults.stream()
        .filter(x -> x.getLevel().equals(ERROR))
        .forEach(x -> LOGGER.error(x + lineSeparator()));

    validationResults.stream()
        .filter(x -> x.getLevel().equals(WARN))
        .forEach(x -> LOGGER.warn(x + lineSeparator()));

    validationResults.stream()
        .filter(x -> x.getLevel().equals(INFO))
        .forEach(x -> LOGGER.warn(x + lineSeparator()));
  }

  public long countResults(ValidationRule.Level level) {
    return validationResults.stream()
        .filter(x -> x.getLevel().equals(level))
        .count();
  }

  public List<ValidationResult> getResults() {
    return validationResults;
  }

  public static final List<ValidationRule> ALL_RULES = new ArrayList<>();
  static {
    ALL_RULES.add(new FileSchemeExistRule());
    ALL_RULES.add(new DescriptorPathMustExistInApiRule());
    ALL_RULES.add(new DescriptorParamsShouldBeTypedRule());
    ALL_RULES.add(new DescriptorParamsMustBeUniquelyTypedRule());
    ALL_RULES.add(new DescriptorOperationMustExistInApiRule());
    ALL_RULES.add(new OperationMustDeclareResponseBodyRule());
    ALL_RULES.add(new AtLeastOneSupportedSecuritySchemeRule());
    ALL_RULES.add(new ConnectorMustHaveRootBaseUriRule());
    ALL_RULES.add(new ExtensionXmlMustBeValidRule());
    ALL_RULES.add(new BaseJavaPackageMustBeValidRule());
    ALL_RULES.add(new GavGroupIdMustBeValidRule());
    ALL_RULES.add(new GavArtifactIdMustBeValidRule());
    ALL_RULES.add(new GavVersionMustBeValidRule());
    ALL_RULES.add(new NotSupportedSecuritySchemeRule());
    ALL_RULES.add(new ConnectorNameMustBeFriendlyRule());
    ALL_RULES.add(new OperationNameMustBeFriendlyRule());
    ALL_RULES.add(new OperationSampleDataAutogeneratedMustBeAccepted());
    ALL_RULES.add(new IdentifiersScriptsMustNotBeEmptyRule());
    ALL_RULES.add(new OperationDisplayNameScriptMustCompileRule());
    ALL_RULES.add(new OperationIdentifierScriptMustCompileRule());
    ALL_RULES.add(new OperationIdentifierScriptMustNotRepeatRule());
    ALL_RULES.add(new OperationInternalNameMustNotRepeatRule());
    ALL_RULES.add(new ParameterIdentifierScriptMustCompileRule());
    ALL_RULES.add(new ParameterInternalNameMustNotRepeatRule());
    ALL_RULES.add(new ParametersIdentifiersMustNotRepeatRule());
    ALL_RULES.add(new BodyIdentifierScriptMustCompileRule());
    ALL_RULES.add(new BodyIdentifierScriptMustNotRepeatRule());
    ALL_RULES.add(new NoDefaultInputMediaTypeDefinedRule());
    ALL_RULES.add(new NoDefaultOutputMediaTypeDefinedRule());
    ALL_RULES.add(new DescriptorParameterMustBePresentInApiRule());
    ALL_RULES.add(new OverrideParameterDescriptorMustBePresentInApiRule());
    ALL_RULES.add(new NoInputMediaTypeDefinedRule());
    ALL_RULES.add(new DescriptorInputMediaTypeMustBePresentInApiRule());
    ALL_RULES.add(new NoOutputMediaTypeDefinedRule());
    ALL_RULES.add(new DescriptorOtputMediaTypeMustBePresentInApiRule());
    ALL_RULES.add(new DescriptorRedundantOutputDefinitionRule());
    ALL_RULES.add(new PagedOperationsOperationMustDeclareResponseBodyRule());
    ALL_RULES.add(new TriggerParameterSetAsMetadataKeyIdCannotHaveValueProviderRule());
    ALL_RULES.add(new TriggerPathAndMethodMustExistInApiOrOperationAdapterRule());
    ALL_RULES.add(new TriggerBindingParameterMustExistInApiRule());
    ALL_RULES.add(new TriggerBaseEndpointMustBeDefinedRule());
    ALL_RULES.add(new TriggerUriParamsMustBeCompleteRule());
    ALL_RULES.add(new TriggerDataWeaveExpressionsMustCompileRule());
    ALL_RULES.add(new ValueProviderReferenceArgumentsExistsRule());
    ALL_RULES.add(new ValueProviderReferenceParametersMustReferencedRule());
    ALL_RULES.add(new ValueProviderReferenceArgumentsHaveValidFormatRule());
    ALL_RULES.add(new SampleDataReferenceArgumentsExistsRule());
    ALL_RULES.add(new SampleDataReferenceParametersMustReferencedRule());
    ALL_RULES.add(new PagedOperationsParameterMustExistRule());
    ALL_RULES.add(new PagedOperationsExpressionMustCompileRule());
    ALL_RULES.add(new DescriptorAdditionalParameterMandatoryFieldsRule());
    ALL_RULES.add(new DescriptorAdditionalParameterMustNotBePresentInApiRule());
    ALL_RULES.add(new ConnectorMustHaveCompleteGavRule());
    ALL_RULES.add(new ValueProviderReferenceArgumentsOperationParametersExistsRule());
    ALL_RULES.add(new TriggerMandatoryFieldsRule());
    ALL_RULES.add(new ValueProviderBodyLevelReferenceArgumentsHaveValidFormatRule());
    ALL_RULES.add(new OperationAdapterRequiredParametersMustNotBeIgnoredRule());
    ALL_RULES.add(new OperationAdapterBindingParameterWithValueMustNotBeIgnoredRule());
    ALL_RULES.add(new PagedParametersMustNotBeIgnoredRule());
    ALL_RULES.add(new OauthAuthorizationUrlMustExistRule());
    ALL_RULES.add(new OauthAccessTokenMustExistRule());
    ALL_RULES.add(new BaseTriggerMustExistRule());
    ALL_RULES.add(new TriggerCircularDependenciesRule());
    ALL_RULES.add(new BaseOperationMustExistRule());
    ALL_RULES.add(new DescriptorReservedWordsRule());
    ALL_RULES.add(new ParametersUsedInOperationAdapterBindingsShouldExistRule());
    ALL_RULES.add(new SecuritySchemeMustBeSupportedByTheApiRule());
    ALL_RULES.add(new SecurityTypeMustBeTheSameRule());
    ALL_RULES.add(new OperationAdapterRequestBindingParametersMustExistsRule());
    ALL_RULES.add(new SampleDataReferenceBindingParametersExistsRule());
    ALL_RULES.add(new ValueProviderReferenceBindingParametersExistsRule());
    ALL_RULES.add(new ValueProviderBodyLevelReferenceBindingParametersExistsRule());
    ALL_RULES.add(new ValueProviderInlineBindingParametersExistsRule());
    ALL_RULES.add(new ValueProviderBodyLevelInlineBindingParametersExistsRule());
    ALL_RULES.add(new SampleDataInlineBindingParametersExistsRule());
    ALL_RULES.add(new ValueProviderMustHaveRequestOrScriptDefinitionRule());
    ALL_RULES.add(new FileInOverrideMustExistInGeneratedCodeRule());
    ALL_RULES.add(new RefinementsDontGoInOverrideFolderRule());
    ALL_RULES.add(new OnlyRefinementsGoInRefinementFolderRule());
    ALL_RULES.add(new SampleDataArgumentsOptionalityValidationRule());
    ALL_RULES.add(new OperationAdapterBodyBindingShouldDefineBoundaryOnMultipartFormDataRule());
    ALL_RULES.add(new RefinedClassesMustBeMarkedInDescriptorFolderRule());
    ALL_RULES.add(new XmlAndJSONSchemasReferencedByDescriptorMustBeWellFormedRule());
    ALL_RULES.add(new SecurityNotSupportedInApiMustHaveAnAssociatedCustomTypeInDescriptor());
    ALL_RULES.add(new DynamicOperationParametersMustNotBeTypedRule());
    ALL_RULES.add(new DynamicTriggerParametersMustNotBeTypedRule());
    ALL_RULES.add(new TestConnectionExpressionMustCompileRule());
    ALL_RULES.add(new RequiredParametersMustNotBeIgnoredRule());
    ALL_RULES.add(new OperationAdapterDifferentOutputMediaTypeRule());
    ALL_RULES.add(new OperationAdapterOutputSchemaWithoutMediaTypeRule());
    ALL_RULES.add(new OperationIdentifierExpressionNotOverrideableRule());
    ALL_RULES.add(new BodyIdentifierExpressionNotOverrideableRule());
    ALL_RULES.add(new ParameterIdentifierExpressionNotOverrideableRule());
    ALL_RULES.add(new MockedSecuritySchemeMustBeUniqueRule());
    ALL_RULES.add(new DescriptorOperationMultipartResponseNotSupportedRule());
    ALL_RULES.add(new TriggerBaseFromFqnNotAllowedRule());
    ALL_RULES.add(new OperationAdapterMaxOneCustomFieldRule());
    ALL_RULES.add(new OperationAdapterMaxOneMetadataKeyRule());
    ALL_RULES.add(new OperationCustomFieldBindingMustCompileRule());
    ALL_RULES.add(new OperationCustomFieldBindingMustBeMetadataKeyRule());
    ALL_RULES.add(new OperationCustomFieldTransformationScriptMustCompileRule());
    ALL_RULES.add(new ConnectorMustHaveOneOperationOrTriggerRule());
    ALL_RULES.add(new TriggerDataWeaveExpressionsShouldReturnExpectedMetadataTypeRule());
  }
}
