/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.operation.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ConnectorModelValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.WARN;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isBlank;

public class ConnectorMustHaveRootBaseUriRule extends ConnectorModelValidationRule {

  public ConnectorMustHaveRootBaseUriRule() {
    super("Base URI cannot be inferred from the API spec.",
          EMPTY,
          WARN);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorModel connectorModel) {
    String baseUri = connectorModel.getBaseUri().getUri();
    if (isBlank(baseUri)) {
      Set<String> baseUris = connectorModel.getOperations().stream()
          .map(ConnectorOperation::getBaseUris)
          .flatMap(List::stream)
          .collect(Collectors.toSet());

      return singletonList(getValidationError(baseUris));
    }

    return emptyList();
  }

  private ValidationResult getValidationError(Set<String> baseUris) {
    String detail;

    if (baseUris.isEmpty()) {
      detail = "No Base URI declared. Please define one.";
    } else {
      detail = "There are more than one base URI, please define only one.";
    }

    return new ValidationResult(this, detail);
  }
}
