/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.operation.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.ValidationConstants;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ConnectorModelValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import org.apache.commons.lang3.StringUtils;

import java.util.List;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.WARN;
import static java.lang.String.format;
import static java.lang.String.join;
import static java.util.stream.Collectors.toList;

public class OperationSampleDataAutogeneratedMustBeAccepted extends ConnectorModelValidationRule {

  public OperationSampleDataAutogeneratedMustBeAccepted() {
    super("Operation example sample data can not be processed",
          "An exception occurred while extracting and processing sample data example to be auto generated.",
          WARN);
  }

  private boolean isResponseExampleAllowedIfExists(ConnectorOperation operation) {
    if (operation.getOutputMetadata() != null && operation.getOutputMetadata().getExample() != null &&
        ValidationConstants.ALLOWED_MEDIA_TYPES_FOR_EXAMPLE.stream().noneMatch(x -> x
            .equals(operation.getOutputMetadata().getMediaType().toString()))) {
      return false;
    }
    return true;
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorModel connectorModel) {
    return connectorModel.getOperations().stream()
        .filter(op -> StringUtils.isBlank(op.getFqn()))
        .filter(x -> !isResponseExampleAllowedIfExists(x))
        .map(this::getValidationError)
        .collect(toList());
  }

  private ValidationResult getValidationError(ConnectorOperation connectorOperation) {

    String detail =
        format("Operation with PATH: %s and METHOD: %s has a based on an example sample data that is not allowed to be auto generated."
            + " Allowed media types are: %s. The type found is: %s",
               connectorOperation.getPath(),
               connectorOperation.getHttpMethod().name(),
               join(",", ValidationConstants.ALLOWED_MEDIA_TYPES_FOR_EXAMPLE),
               connectorOperation.getOutputMetadata().getMediaType().toString());

    return new ValidationResult(this, detail);
  }
}
