/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel;

import com.mulesoft.connectivity.rest.commons.api.dw.DWBindings;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.SchemaInferenceService;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.generic.ParameterDataType;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.TriggerParameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.trigger.Trigger;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ConnectorModelValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.util.DataWeaveExpressionValidationUtils;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.ObjectTypeBuilder;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.impl.DefaultArrayType;
import org.mule.metadata.api.model.impl.DefaultStringType;
import org.mule.metadata.api.model.impl.DefaultNumberType;
import org.mule.metadata.api.model.impl.DefaultDateTimeType;
import org.mule.metadata.api.model.impl.DefaultDateType;
import org.mule.metadata.api.model.impl.DefaultLocalTimeType;
import org.mule.metadata.api.model.impl.DefaultTimeType;
import org.mule.metadata.api.model.impl.DefaultTimeZoneType;
import org.mule.metadata.api.model.impl.DefaultLocalDateTimeType;

import java.util.Optional;
import java.util.Map;
import java.util.List;
import java.util.ArrayList;
import java.util.HashMap;

import static com.mulesoft.connectivity.rest.commons.internal.metadatamodel.RestJsonTypeLoader.JSON;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.DataWeaveExpressionValidationUtils.getMetadataTypeForTypeName;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.WARN;

public class TriggerDataWeaveExpressionsShouldReturnExpectedMetadataTypeRule extends ConnectorModelValidationRule {

  public TriggerDataWeaveExpressionsShouldReturnExpectedMetadataTypeRule() {
    super("Trigger DataWeave expression is incorrect.",
          "DataWeave Expressions evaluation should match with the expected type.", WARN);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorModel connectorModel) {
    List<ValidationResult> validationResults = new ArrayList<>();
    connectorModel.getTriggers().stream().forEach(trigger -> validateTriggerExpressions(trigger, validationResults));
    return validationResults;
  }

  private void validateTriggerExpressions(Trigger trigger, List<ValidationResult> validationResults) {
    validateTriggerItemsExpression(trigger, validationResults);
    validateTriggerWatermarkExpression(trigger, validationResults);
    validateTriggerIdentityExpression(trigger, validationResults);
  }

  private void validateTriggerItemsExpression(Trigger trigger, List<ValidationResult> validationResults) {
    BindingsBuilder bindings = getBaseTriggerBindings(trigger);

    Optional<ValidationResult> itemsValidationResult =
        DataWeaveExpressionValidationUtils.validateExpression(trigger.getItemsExpression(), bindings.build(),
                                                              new Class[] {DefaultArrayType.class},
                                                              trigger.getName(),
                                                              this);
    if (itemsValidationResult.isPresent()) {
      validationResults.add(itemsValidationResult.get());
    }
  }


  private void validateTriggerWatermarkExpression(Trigger trigger, List<ValidationResult> validationResults) {
    BindingsBuilder bindings = getBaseTriggerBindings(trigger);
    bindings.item(trigger.getTriggerOutputType().getTypeSchema().getRawSchema());

    Optional<ValidationResult> itemsValidationResult =
        DataWeaveExpressionValidationUtils.validateExpression(trigger.getWatermarkExpression(), bindings.build(),
                                                              new Class[] {DefaultStringType.class, DefaultNumberType.class,
                                                                  DefaultDateTimeType.class, DefaultDateType.class,
                                                                  DefaultLocalDateTimeType.class, DefaultLocalTimeType.class,
                                                                  DefaultTimeType.class, DefaultTimeZoneType.class},
                                                              trigger.getName(),
                                                              this);
    if (itemsValidationResult.isPresent()) {
      validationResults.add(itemsValidationResult.get());
    }
  }

  private void validateTriggerIdentityExpression(Trigger trigger, List<ValidationResult> validationResults) {
    BindingsBuilder bindings = getBaseTriggerBindings(trigger);
    bindings.item(trigger.getTriggerOutputType().getTypeSchema().getRawSchema());

    Optional<ValidationResult> itemsValidationResult =
        DataWeaveExpressionValidationUtils.validateExpression(trigger.getIdentityExpression(), bindings.build(),
                                                              new Class[] {DefaultStringType.class, DefaultNumberType.class},
                                                              trigger.getName(),
                                                              this);
    if (itemsValidationResult.isPresent()) {
      validationResults.add(itemsValidationResult.get());
    }
  }

  private BindingsBuilder getBaseTriggerBindings(Trigger trigger) {
    BindingsBuilder bindings = new BindingsBuilder();
    bindings.payload(trigger.getOperation().getOutputMetadata().getTypeSchema().getRawSchema());
    bindings.responseBody(trigger.getOperation().getOutputMetadata().getTypeSchema().getRawSchema());
    bindings.parameters(trigger.getParameters());
    bindings.watermark(trigger.getWatermarkType());
    return bindings;
  }

  public static class BindingsBuilder {

    private MetadataType payload;
    private MetadataType responseBody;
    private MetadataType parameters;
    private MetadataType watermark;
    private MetadataType item;
    private BaseTypeBuilder baseTypeBuilder = BaseTypeBuilder.create(JSON);

    public TriggerDataWeaveExpressionsShouldReturnExpectedMetadataTypeRule.BindingsBuilder payload(String schema) {
      this.payload = SchemaInferenceService.getSchemaMetadataType(schema);
      return this;
    }

    public TriggerDataWeaveExpressionsShouldReturnExpectedMetadataTypeRule.BindingsBuilder responseBody(String schema) {
      ObjectTypeBuilder response = baseTypeBuilder.objectType();
      response.addField().key(DWBindings.BODY.getBinding()).value(SchemaInferenceService.getSchemaMetadataType(schema));
      this.responseBody = response.build();
      return this;
    }

    public TriggerDataWeaveExpressionsShouldReturnExpectedMetadataTypeRule.BindingsBuilder parameters(List<TriggerParameter> triggerParameters) {
      ObjectTypeBuilder parameters = baseTypeBuilder.objectType();
      triggerParameters.forEach(parameter -> {
        parameters.addField().key(parameter.getInternalName()).value(getMetadataTypeForTypeName(parameter.getType()));
      });
      this.parameters = parameters.build();
      return this;
    }

    public TriggerDataWeaveExpressionsShouldReturnExpectedMetadataTypeRule.BindingsBuilder watermark(ParameterDataType watermarkType) {
      this.watermark = getMetadataTypeForTypeName(watermarkType);
      return this;
    }

    public TriggerDataWeaveExpressionsShouldReturnExpectedMetadataTypeRule.BindingsBuilder item(String schema) {
      this.item = SchemaInferenceService.getSchemaMetadataType(schema);
      return this;
    }

    public Map<String, MetadataType> build() {
      Map<String, MetadataType> bindings = new HashMap<>();
      if (payload != null) {
        bindings.put("payload", payload);
      }
      if (responseBody != null) {
        bindings.put("response", responseBody);
      }
      if (parameters != null) {
        bindings.put("parameters", parameters);
      }
      if (watermark != null) {
        bindings.put("watermark", watermark);
      }
      if (item != null) {
        bindings.put("item", item);
      }
      return bindings;
    }
  }
}

