/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.common.ArgumentDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.dataexpressions.HttpRequestDataExpressionBindingDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.dataexpressions.HttpRequestDataExpressionDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.resolvers.ResolverDefinitionDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.resolvers.ResolverReferenceDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIParameterModel;

import java.util.ArrayList;
import java.util.List;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.QUERY;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.URI;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.HEADER;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.WARN;
import static java.lang.String.format;
import static org.apache.commons.lang3.StringUtils.EMPTY;

public abstract class AbstractResolverReferenceBindingParametersExistsRule<T extends ResolverDefinitionDescriptor<T>>
    extends DescriptorValidationRule {

  private final String ERROR_TEMPLATE;

  public AbstractResolverReferenceBindingParametersExistsRule(String resolverType) {
    super(format("Parameter bindings defined in a %s must exist in the API spec.", resolverType), EMPTY, WARN);
    ERROR_TEMPLATE = new StringBuilder()
        .append("%s parameter binding '%s' defined in ")
        .append(resolverType)
        .append(" with name: %s declares a parameter binding that does not exist in the API spec.")
        .toString();
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {
    final List<ValidationResult> results = new ArrayList<>();

    for (ResolverReferenceDescriptor<T> reference : getResolverReferenceDescriptors(connectorDescriptor)) {
      HttpRequestDataExpressionDescriptor resolverDescriptor = getResolverDescriptor(reference);
      if (resolverDescriptor == null) {
        continue;
      }
      HttpRequestDataExpressionBindingDescriptor bindings = resolverDescriptor.getBindings();
      String path = resolverDescriptor.getPath();
      String method = resolverDescriptor.getMethod().getName();

      APIOperationModel operationModel = apiModel.findOperation(path, method).orElse(null);
      if (operationModel == null) {
        continue;
      }

      if (bindings != null) {
        analyzeParameters(results, reference, operationModel.getHeadersModel(), bindings.getHeaders(), HEADER);
        analyzeParameters(results, reference, operationModel.getQueryParamsModel(), bindings.getQueryParameters(), QUERY);
        analyzeParameters(results, reference, operationModel.getUriParamsModel(), bindings.getUriParameters(), URI);
      }
    }
    return results;
  }

  private void analyzeParameters(List<ValidationResult> results, ResolverReferenceDescriptor<T> reference,
                                 List<APIParameterModel> baseParameters, List<ArgumentDescriptor> bindings,
                                 ParameterType bindingParameterType) {
    if (bindings == null) {
      return;
    }

    for (ArgumentDescriptor binding : bindings) {
      if (baseParameters.isEmpty()
          || baseParameters.stream().noneMatch(x -> x.getExternalName().equals(binding.getName()))) {
        results.add(getValidationError(bindingParameterType, binding, reference));
      }
    }

  }

  protected abstract List<ResolverReferenceDescriptor<T>> getResolverReferenceDescriptors(ConnectorDescriptor connectorDescriptor);

  protected abstract HttpRequestDataExpressionDescriptor getResolverDescriptor(ResolverReferenceDescriptor<T> reference);

  private ValidationResult getValidationError(ParameterType parameterType, ArgumentDescriptor binding,
                                              ResolverReferenceDescriptor<T> reference) {
    return new ValidationResult(this,
                                format(ERROR_TEMPLATE, parameterType.getName(), binding.getName(),
                                       reference.getDeclaration().getName()),
                                reference.getDeclaration().getLocation());
  }


}
