/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.HEADER;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.QUERY;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.URI;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.ComparisonUtil;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.EndPointDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIParameterModel;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

public class DescriptorParameterMustBePresentInApiRule extends DescriptorValidationRule {

  // R016
  public DescriptorParameterMustBePresentInApiRule() {
    super("Operation parameters declared in the connector descriptor must be present in the API spec",
          EMPTY,
          ERROR);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {

    List<ValidationResult> results = new ArrayList<>();

    for (EndPointDescriptor endPointDescriptor : connectorDescriptor.getEndpoints()) {
      for (OperationDescriptor operationDescriptor : endPointDescriptor.getOperations()) {
        if (operationDescriptor.getExpects() != null) {
          results.addAll(validateParametersExists(operationDescriptor.getExpects().getUriParameter(), URI,
                                                  endPointDescriptor.getPath(), operationDescriptor.getMethod(), apiModel));

          results.addAll(validateParametersExists(operationDescriptor.getExpects().getQueryParameter(), QUERY,
                                                  endPointDescriptor.getPath(), operationDescriptor.getMethod(), apiModel));

          results.addAll(validateParametersExists(operationDescriptor.getExpects().getHeader(), HEADER,
                                                  endPointDescriptor.getPath(), operationDescriptor.getMethod(), apiModel));

        }
      }
    }

    return results;
  }

  private List<ValidationResult> validateParametersExists(List<ParameterDescriptor> parameters,
                                                          ParameterType parameterType,
                                                          String path,
                                                          String method,
                                                          APIModel apiModel) {
    return parameters.stream()
        .filter(paramDesc -> !paramDesc.isAdditional())
        .map(paramDesc -> validateParameterExists(paramDesc, parameterType, path, method, apiModel))
        .filter(Objects::nonNull)
        .collect(toList());
  }

  private ValidationResult validateParameterExists(ParameterDescriptor parameter,
                                                   ParameterType parameterType,
                                                   String path,
                                                   String method,
                                                   APIModel apiModel) {

    return apiModel.findOperation(path, method)
        .map(x -> validateParameterExists(x, parameter, parameterType))
        .orElse(null);
  }

  private ValidationResult validateParameterExists(APIOperationModel apiOperation, ParameterDescriptor parameter,
                                                   ParameterType parameterType) {
    return apiOperation.getParameter(parameterType.getApiParameterType(), parameter.getParamName()).isPresent()
        ? null
        : getValidationError(apiOperation, parameter, parameterType);
  }

  private ValidationResult getValidationError(APIOperationModel apiOperationModel, ParameterDescriptor paramDescriptor,
                                              ParameterType parameterType) {
    String detail =
        "API Operation with PATH: "
            + apiOperationModel.getPath() +
            " and METHOD: "
            + apiOperationModel.getHttpMethod().toUpperCase() +
            " does not declare a "
            + parameterType.getName()
            + " named '"
            + paramDescriptor.getParamName() + "'";

    return new ValidationResult(this, detail, paramDescriptor.getLocation());
  }
}
