/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.AuxiliarParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.EndPointDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationAdapterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.RequestDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static java.lang.String.format;
import static org.apache.commons.lang3.StringUtils.EMPTY;

public class DescriptorReservedWordsRule extends DescriptorValidationRule {


  /*
   * These words are reserved from the sdk. They were tested taking into account the following class:
   * https://github.com/mulesoft/mule-extensions-api/blob/master/mule-extensions-api/src/main/java/org/mule/runtime/extension/api/
   * ExtensionConstants.java
   */
  static final List<String> RESERVED_WORDS = Arrays.asList("targetValue", "target", "streamingStrategy", "reconnectionStrategy");

  public DescriptorReservedWordsRule() {
    super("The custom operations must not have sdk reserved words in their parameter declarations.",
          EMPTY,
          ERROR);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {
    final List<ValidationResult> validationResults = new ArrayList<>();
    for (OperationAdapterDescriptor operationAdapter : connectorDescriptor.getOperationAdapterDescriptors()) {
      for (AuxiliarParameterDescriptor parameter : operationAdapter.getParameters()) {
        if (RESERVED_WORDS.contains(parameter.getName())) {
          validationResults.add(getOperationAdapterValidationError(operationAdapter, parameter));
        }
      }
    }


    for (EndPointDescriptor endPointDescriptor : connectorDescriptor.getEndpoints()) {
      for (OperationDescriptor operationDescriptor : endPointDescriptor.getOperations()) {
        List<ParameterDescriptor> parameters = getAllParameters(operationDescriptor.getExpects());
        for (ParameterDescriptor parameter : parameters) {
          if (RESERVED_WORDS.contains(parameter.getParameterIdentifier()))
            validationResults
                .add(getParameterIdentiferValidationError(parameter, operationDescriptor, endPointDescriptor.getPath()));
        }
      }
    }


    return validationResults;
  }

  private ValidationResult getOperationAdapterValidationError(OperationAdapterDescriptor operationAdapterDescriptor,
                                                              AuxiliarParameterDescriptor auxiliarParameterDescriptor) {

    String detail =
        format("The operation: " + operationAdapterDescriptor.getOperationId() + " has a parameter declared with name: "
            + auxiliarParameterDescriptor.getName() + ", this name cannot be used because it is a reserved word.");
    return new ValidationResult(this, detail, operationAdapterDescriptor.getLocation());

  }

  private ValidationResult getParameterIdentiferValidationError(ParameterDescriptor parameterDescriptor,
                                                                OperationDescriptor operationDescriptor,
                                                                String path)

  {
    String detail =
        format("The operation with path: " + path + " has a parameter declared with parameterIdentifer: "
            + parameterDescriptor.getParameterIdentifier() + ", this name cannot be used because it is a reserved word.");
    return new ValidationResult(this, detail, operationDescriptor.getLocation());
  }

  private List<ParameterDescriptor> getAllParameters(RequestDescriptor requestDescriptor) {
    List<ParameterDescriptor> parameters = new ArrayList<>();

    if (requestDescriptor != null) {
      parameters.addAll(requestDescriptor.getUriParameter());
      parameters.addAll(requestDescriptor.getQueryParameter());
      parameters.addAll(requestDescriptor.getHeader());
    }

    return parameters;
  }
}
