/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.WARN;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.getApiOperation;
import static org.apache.commons.lang3.StringUtils.EMPTY;

import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.AuxiliarBodyBindingDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationAdapterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;

import java.util.ArrayList;
import java.util.List;

public class OperationAdapterDifferentOutputMediaTypeRule extends DescriptorValidationRule {

  public OperationAdapterDifferentOutputMediaTypeRule() {
    super("Operation response media type is different from the Api.", EMPTY, WARN);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {
    final List<ValidationResult> results = new ArrayList<>();

    List<OperationAdapterDescriptor> operationAdapterDescriptors = connectorDescriptor.getOperationAdapterDescriptors();
    for (OperationAdapterDescriptor operationAdapterDescriptor : operationAdapterDescriptors) {
      if (operationAdapterDescriptor.getResponseBindings() != null
          && operationAdapterDescriptor.getResponseBindings().getBodyBinding() != null) {
        AuxiliarBodyBindingDescriptor bodyBinding = operationAdapterDescriptor.getResponseBindings().getBodyBinding();
        if (bodyBinding.getContentType() != null && !hasSameMediaType(apiModel, operationAdapterDescriptor, bodyBinding)) {
          results.add(getValidationError(operationAdapterDescriptor, bodyBinding));
        }
      }
    }

    return results;
  }

  private boolean hasSameMediaType(APIModel apiModel, OperationAdapterDescriptor operationAdapterDescriptor,
                                   AuxiliarBodyBindingDescriptor bodyBinding) {
    APIOperationModel apiOperationModel = getApiOperation(apiModel, operationAdapterDescriptor);
    return apiOperationModel.getOutputMetadataModel().stream()
        .anyMatch(m -> m.getMediaType().toString().equalsIgnoreCase(bodyBinding.getContentType()));
  }

  private ValidationResult getValidationError(OperationAdapterDescriptor operationAdapterDescriptor,
                                              AuxiliarBodyBindingDescriptor bodyBinding) {
    String detail = "Operation with name "
        + operationAdapterDescriptor.getOperationId()
        + " declares a mediaType different from the one defined in the Api.";

    return new ValidationResult(this, detail, bodyBinding.getLocation());
  }

}
