/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static org.apache.commons.lang3.StringUtils.EMPTY;

import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.AuxiliarBodyBindingDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.AuxiliarParameterResponseBindingsDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DescriptorElementLocation;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationAdapterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OverridesBodyDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OverridesRequestDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.ArrayList;
import java.util.List;

public class OperationAdapterMaxOneCustomFieldRule extends DescriptorValidationRule {

  public OperationAdapterMaxOneCustomFieldRule() {
    super("Operation declares more than one custom field.", EMPTY, ERROR);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {
    final List<ValidationResult> results = new ArrayList<>();

    for (OperationAdapterDescriptor operationAdapterDescriptor : connectorDescriptor.getOperationAdapterDescriptors()) {
      validateRequestBodyCustomFeilds(operationAdapterDescriptor, results);
      validateResponseBodyCustomFeilds(operationAdapterDescriptor, results);
    }

    return results;
  }

  private void validateResponseBodyCustomFeilds(OperationAdapterDescriptor operationAdapterDescriptor,
                                                List<ValidationResult> results) {
    if (operationAdapterDescriptor.getBaseEndpoint() != null
        && operationAdapterDescriptor.getBaseEndpoint().getOverrides() != null) {
      OverridesBodyDescriptor overridesBody = operationAdapterDescriptor.getBaseEndpoint().getOverrides().getBody();
      if (overridesBody != null) {
        int numberOfCustomFields = 0;
        if (overridesBody.getOverrideResolver() != null) {
          numberOfCustomFields++;
        }
        numberOfCustomFields +=
            overridesBody.getFieldDescriptors().stream().filter(field -> field.getOverrideResolver() != null).count();
        if (numberOfCustomFields > 1) {
          results.add(getValidationError(operationAdapterDescriptor, "response", overridesBody.getLocation()));
        }
      }
    }
  }

  private void validateRequestBodyCustomFeilds(OperationAdapterDescriptor operationAdapterDescriptor,
                                               List<ValidationResult> results) {
    AuxiliarParameterResponseBindingsDescriptor responseBindings = operationAdapterDescriptor.getResponseBindings();
    if (responseBindings != null) {
      AuxiliarBodyBindingDescriptor overridesBody = responseBindings.getBodyBinding();
      if (overridesBody != null) {
        int numberOfCustomFields = 0;
        if (overridesBody.getOverrideResolver() != null) {
          numberOfCustomFields++;
        }
        numberOfCustomFields +=
            overridesBody.getAuxiliarBodyField().stream().filter(field -> field.getOverrideResolver() != null).count();
        if (numberOfCustomFields > 1) {
          results.add(getValidationError(operationAdapterDescriptor, "request", overridesBody.getLocation()));
        }
      }
    }
  }

  private ValidationResult getValidationError(OperationAdapterDescriptor operationAdapterDescriptor,
                                              String location,
                                              DescriptorElementLocation descriptorElementLocation) {
    String detail = "Operation with name "
        + operationAdapterDescriptor.getOperationId()
        + " declares more than one custom field in the "
        + location
        + " body.";

    return new ValidationResult(this, detail, descriptorElementLocation);
  }

}
