/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.HEADER;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.QUERY;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.URI;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;

import com.mulesoft.connectivity.rest.commons.api.dw.DWBindings;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.ComparisonUtil;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DescriptorElementLocation;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationAdapterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OverrideResolverDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OverridesParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OverridesRequestDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.common.ArgumentDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.dataexpressions.HttpRequestDataExpressionBindingDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.commons.lang3.tuple.Pair;

public class OperationCustomFieldBindingMustBeMetadataKeyRule extends AbstractCustomFieldRule {

  public OperationCustomFieldBindingMustBeMetadataKeyRule() {
    super("Parameters used in a custom field binding expression must be Metadata Key");
  }


  @Override
  protected void validateOverrideResolver(OperationAdapterDescriptor operationAdapterDescriptor,
                                          OverrideResolverDescriptor overrideResolver,
                                          List<ValidationResult> results) {
    HttpRequestDataExpressionBindingDescriptor bindings = overrideResolver.getRequest().getBindings();

    if (bindings != null) {
      Map<String, List<String>> metadataKey = findMetadataKey(operationAdapterDescriptor);
      bindings.getQueryParameters().forEach(param -> validateBinding(operationAdapterDescriptor, param, metadataKey, results));
      bindings.getUriParameters().forEach(param -> validateBinding(operationAdapterDescriptor, param, metadataKey, results));
      bindings.getHeaders().forEach(param -> validateBinding(operationAdapterDescriptor, param, metadataKey, results));
    }

  }

  private void validateBinding(OperationAdapterDescriptor operationAdapterDescriptor,
                               ArgumentDescriptor param,
                               Map<String, List<String>> metadataKeys,
                               List<ValidationResult> results) {
    Pair<String, String> parameterTypeAndName = findParameterTypeAndName(param.getValue().getExpression());
    if (parameterTypeAndName != null
        && !metadataKeyExists(parameterTypeAndName.getLeft(), parameterTypeAndName.getRight(), metadataKeys)) {
      results.add(getValidationError(operationAdapterDescriptor, parameterTypeAndName.getLeft(), parameterTypeAndName.getRight(),
                                     param.getLocation()));
    }
  }

  private boolean metadataKeyExists(String parameterType, String parameterName, Map<String, List<String>> metadataKeys) {
    return metadataKeys.containsKey(parameterType) &&
        metadataKeys.get(parameterType)
            .stream()
            .anyMatch(mk -> ComparisonUtil.externalNameParamsComparison(parameterName, mk,
                                                                        ParameterType.fromBinding(parameterType)));

  }

  private Pair<String, String> findParameterTypeAndName(String expression) {
    String parameterType = getParameterTypeReference(expression);
    if (parameterType == null) {
      return null;
    }
    String parameterName;
    if (parameterType.equals(DWBindings.REQUEST.getBinding())) {
      parameterType = getParameterReference(expression, emptyList());
      parameterName = getParameterReference(expression, singletonList(parameterType));
    } else {
      parameterName = getParameterReference(expression, emptyList());
    }

    return Pair.of(BINDINGS_MAPPING.getOrDefault(parameterType, parameterType), parameterName);
  }

  private Map<String, List<String>> findMetadataKey(OperationAdapterDescriptor operationAdapterDescriptor) {
    if (operationAdapterDescriptor.getBaseEndpoint() == null
        || operationAdapterDescriptor.getBaseEndpoint().getOverrides() == null) {
      return emptyMap();
    }

    OverridesRequestDescriptor overrides = operationAdapterDescriptor.getBaseEndpoint().getOverrides();
    Map<String, List<String>> metadataKeys = new HashMap<>();
    metadataKeys.put(QUERY.getBinding(), findMetadataKey(overrides.getQueryParameter()));
    metadataKeys.put(URI.getBinding(), findMetadataKey(overrides.getUriParameter()));
    metadataKeys.put(HEADER.getBinding(), findMetadataKey(overrides.getHeader()));
    return metadataKeys;
  }

  private List<String> findMetadataKey(List<OverridesParameterDescriptor> parameters) {
    return parameters.stream()
        .filter(param -> param.getMetadataKey() != null)
        .map(OverridesParameterDescriptor::getParamName)
        .collect(Collectors.toList());
  }

  private ValidationResult getValidationError(OperationAdapterDescriptor operationAdapterDescriptor,
                                              String parameterType,
                                              String parameterName,
                                              DescriptorElementLocation descriptorElementLocation) {
    String detail = "Operation with name '"
        + operationAdapterDescriptor.getOperationId()
        + "' declares a custom field resolver request with a binding that uses '"
        + parameterType
        + "' parameter '"
        + parameterName
        + "' that is not Metadata Key.";

    return new ValidationResult(this, detail, descriptorElementLocation);
  }

}
