/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import static java.util.Collections.emptyList;
import static org.apache.commons.lang3.StringUtils.isBlank;

import org.mule.weave.v2.parser.phase.CompilationException;

import com.mulesoft.connectivity.rest.commons.api.dw.DWBindings;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.DataWeaveExpressionParser;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.ExpressionHandlerUtils;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationAdapterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OverrideResolverDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.common.ArgumentDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.dataexpressions.HttpRequestDataExpressionBindingDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;

import java.util.List;

public class OperationCustomFieldBindingMustCompileRule extends AbstractCustomFieldRule {

  public OperationCustomFieldBindingMustCompileRule() {
    super("Custom field bindings expressions must respect the valid format (i.e. request.query.X, request.path.X, request.header.X)");
  }

  @Override
  protected void validateOverrideResolver(OperationAdapterDescriptor operationAdapterDescriptor,
                                          OverrideResolverDescriptor overrideResolver,
                                          List<ValidationResult> results) {
    HttpRequestDataExpressionBindingDescriptor bindings = overrideResolver.getRequest().getBindings();

    if (bindings != null) {
      bindings.getQueryParameters().forEach(param -> validateBinding(operationAdapterDescriptor, param, results));
      bindings.getUriParameters().forEach(param -> validateBinding(operationAdapterDescriptor, param, results));
      bindings.getHeaders().forEach(param -> validateBinding(operationAdapterDescriptor, param, results));
    }

  }

  private void validateBinding(OperationAdapterDescriptor operationAdapterDescriptor,
                               ArgumentDescriptor param,
                               List<ValidationResult> results) {
    if (!isValidFormat(param.getValue().getExpression())) {
      results.add(getValidationError(operationAdapterDescriptor, param));
    }
  }

  private boolean isValidFormat(String value) {
    if (isBlank(value)) {
      return false;
    }
    try {
      ExpressionHandlerUtils.compileDataWeaveScript(value, BINDINGS);
      if (DataWeaveExpressionParser.isBindingUsed(value, DWBindings.REQUEST.getBinding())) {
        String parameterType = getParameterReference(value, emptyList());
        return BINDINGS_MAPPING.containsKey(parameterType);
      }
      return true;
    } catch (CompilationException e) {
      return false;
    }
  }

  private ValidationResult getValidationError(OperationAdapterDescriptor operationAdapterDescriptor,
                                              ArgumentDescriptor param) {
    String detail = "Operation with name '"
        + operationAdapterDescriptor.getOperationId()
        + "' declares a custom field resolver request with a binding expression '"
        + param.getValue().getExpression()
        + "' that does not match the expected format.";

    return new ValidationResult(this, detail, param.getLocation());
  }

}
