/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.distinctByKey;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APISecuritySchemeType.CUSTOM;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APISecuritySchemeType.NOT_SUPPORTED;
import static java.util.stream.Collectors.toList;

import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DescriptorElementLocation;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DescriptorSecurityKind;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.SecuritySchemeBaseDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ApiDescriptorsValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APISecuritySchemeModel;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

public class SecurityNotSupportedInApiMustHaveAnAssociatedCustomTypeInDescriptor extends ApiDescriptorsValidationRule {

  public SecurityNotSupportedInApiMustHaveAnAssociatedCustomTypeInDescriptor() {
    super("Unsupported security schemes need to be customized in the connector descriptor.",
          "Use 'custom' type in the connector descriptor to add a custom implementation.",
          ERROR);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, List<ConnectorDescriptor> connectorDescriptors) {
    List<ValidationResult> validationsResults = new ArrayList<>();

    List<APISecuritySchemeModel> securitySchemeModels = apiModel.getOperationsModel().stream()
        .map(APIOperationModel::getSecuritySchemesModel)
        .flatMap(List::stream)
        .filter(distinctByKey(APISecuritySchemeModel::getName))
        .collect(toList());

    if (securitySchemeModels.stream()
        .allMatch(securitySchemeModel -> securitySchemeModel.getSecuritySchemeType().equals(NOT_SUPPORTED)
            || securitySchemeModel.getSecuritySchemeType().equals(CUSTOM))) {
      for (APISecuritySchemeModel securitySchemeModel : securitySchemeModels) {
        if (securitySchemeModel.getSecuritySchemeType().equals(NOT_SUPPORTED)
            || securitySchemeModel.getSecuritySchemeType().equals(CUSTOM)) {
          Optional<SecuritySchemeBaseDescriptor> securitySchemeDescriptor =
              findSecuritySchemeDescriptor(connectorDescriptors, securitySchemeModel.getName());
          if (securitySchemeDescriptor.isPresent()) {
            if (!securitySchemeDescriptor.get().getKind().equals(DescriptorSecurityKind.CUSTOM)) {
              validationsResults.add(getValidationError(securitySchemeModel, securitySchemeDescriptor.get()));
            }
          } else {
            validationsResults.add(getIsNotSupportedError(securitySchemeModel));
          }
        }
      }
    }


    return validationsResults;
  }

  private Optional<SecuritySchemeBaseDescriptor> findSecuritySchemeDescriptor(List<ConnectorDescriptor> connectorDescriptors,
                                                                              String name) {
    return connectorDescriptors.stream()
        .map(ConnectorDescriptor::getSecurity)
        .flatMap(List::stream)
        .filter(securityScheme -> securityScheme.getName().equals(name)).findFirst();
  }

  private ValidationResult getValidationError(APISecuritySchemeModel apiSecuritySchemeModel,
                                              SecuritySchemeBaseDescriptor securitySchemeDescriptor) {
    String detail =
        "API Security scheme: "
            + "'" + apiSecuritySchemeModel.getName() + "'" +
            " was declared in the descriptor as type: "
            + "'" + securitySchemeDescriptor.getKind() + "'"
            + ". Only 'custom' is supported.";

    return new ValidationResult(this, detail, securitySchemeDescriptor.getLocation());
  }

  private ValidationResult getIsNotSupportedError(APISecuritySchemeModel apiSecuritySchemeModel) {
    String detail =
        "API Security scheme with name: "
            + apiSecuritySchemeModel.getName() +
            " is not supported nor customized in the connector descriptor.";

    return new ValidationResult(this, detail);
  }

}
