/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.ConnectorModelBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.TriggerBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.MethodPathBaseEndpointDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationIdBaseEndpointDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TriggerDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorConnectorModelBuilderValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.List;
import java.util.Optional;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isEmpty;

public class TriggerBaseEndpointMustBeDefinedRule extends DescriptorConnectorModelBuilderValidationRule {

  public TriggerBaseEndpointMustBeDefinedRule() {
    super("Trigger BaseEndpoint must be defined in non native triggers, even by setting its METHOD/PATH or referring its Base operation Identifier.",
          EMPTY, ERROR);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor,
                                         ConnectorModelBuilder connectorModelBuilder) {

    List<TriggerDescriptor> triggerDescriptors = connectorDescriptor.getTriggers();
    if (triggerDescriptors != null && !triggerDescriptors.isEmpty()) {
      return triggerDescriptors.stream()
          .filter(x -> isEmpty(x.getFqn()) && !hasBaseEndpointDefined(apiModel, x, connectorModelBuilder))
          .map(this::getValidationError)
          .collect(toList());
    }

    return emptyList();
  }

  private boolean hasBaseEndpointDefined(APIModel apiModel, TriggerDescriptor triggerDescriptor,
                                         ConnectorModelBuilder connectorModelBuilder) {

    if (triggerDescriptor.getBaseEndpoint() == null) {
      // may be an L3 trigger descriptor, verify if a parent defines its baseEndpoint
      Optional<TriggerBuilder> parentTriggerBuilder = connectorModelBuilder.getTriggerBuilders()
          .stream()
          .filter(x -> x.getMethod() != null
              && !isEmpty(x.getPath())
              && x.getName().equals(x.getName()))
          .findFirst();
      if (parentTriggerBuilder.isPresent()) {
        return apiModel.findOperation(parentTriggerBuilder.get().getPath(), parentTriggerBuilder.get().getMethod().name())
            .isPresent();
      }
      return false;
    }
    if (triggerDescriptor.getBaseEndpoint() instanceof OperationIdBaseEndpointDescriptor) {
      return !isEmpty(((OperationIdBaseEndpointDescriptor) triggerDescriptor.getBaseEndpoint()).getOperationId());
    } else {
      MethodPathBaseEndpointDescriptor descriptorBaseEndpoint =
          (MethodPathBaseEndpointDescriptor) triggerDescriptor.getBaseEndpoint();
      return !isEmpty(descriptorBaseEndpoint.getMethod()) && !isEmpty(descriptorBaseEndpoint.getPath());
    }
  }

  private ValidationResult getValidationError(TriggerDescriptor triggerDescriptor) {
    String detail =
        "Trigger with name: '"
            + triggerDescriptor.getName()
            + "' has incomplete its BaseEndpointDefinition";

    return new ValidationResult(this, detail, triggerDescriptor.getLocation());
  }

}
