/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static java.lang.String.format;

import org.mule.weave.v2.parser.phase.CompilationException;

import com.mulesoft.connectivity.rest.commons.api.dw.DWBindings;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.ExpressionHandlerUtils;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TriggerDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.ArrayList;
import java.util.List;

public class TriggerDataWeaveExpressionsMustCompileRule extends DescriptorValidationRule {

  private static final String[] IMPLICIT_INPUTS_EVENT =
      {"item"};
  private static final String[] IMPLICIT_INPUTS_ITEMS =
      {"parameters", "customParameters", "connection", "configuration", "identity", "payload", DWBindings.RESPONSE.getBinding(),
          "startValue",
          "watermark"};
  private static final String[] IMPLICIT_INPUTS_GENERAL =
      {"parameters", "customParameters", "connection", "configuration", "identity", "payload", DWBindings.RESPONSE.getBinding(),
          "startValue",
          "watermark", "item"};

  public TriggerDataWeaveExpressionsMustCompileRule() {
    super("Trigger DataWeave expression is incorrect.",
          "DataWeave Expressions must compile.", ERROR);
  }

  // validates only triggers that are not ignored

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {

    List<ValidationResult> validationResults = new ArrayList<>();

    connectorDescriptor.getTriggers().stream()
        .filter(triggerDescriptor -> triggerDescriptor.getIgnored() == null || (triggerDescriptor.getIgnored() != null
            && !triggerDescriptor.getIgnored()))
        .forEach(triggerDescriptor -> {
          if (triggerDescriptor.getEventExpression() != null) {
            validateTriggerDWExpression(triggerDescriptor, triggerDescriptor.getEventExpression(), IMPLICIT_INPUTS_EVENT,
                                        "event", validationResults);
          }
          if (triggerDescriptor.getItemsExpression() != null) {
            validateTriggerDWExpression(triggerDescriptor, triggerDescriptor.getItemsExpression(), IMPLICIT_INPUTS_ITEMS,
                                        "items", validationResults);
          }
          if (triggerDescriptor.getIdentityExpression() != null) {
            validateTriggerDWExpression(triggerDescriptor, triggerDescriptor.getIdentityExpression(),
                                        IMPLICIT_INPUTS_GENERAL,
                                        "identity", validationResults);
          }
          if (triggerDescriptor.getWatermark() != null && triggerDescriptor.getWatermark().getValue() != null) {
            validateTriggerDWExpression(triggerDescriptor, triggerDescriptor.getWatermark().getValue(),
                                        IMPLICIT_INPUTS_GENERAL,
                                        "watermark", validationResults);
          }
          if (triggerDescriptor.getStartValueExpression() != null && triggerDescriptor.getStartValueExpression() != null) {
            validateTriggerDWExpression(triggerDescriptor, triggerDescriptor.getStartValueExpression(),
                                        IMPLICIT_INPUTS_GENERAL,
                                        "startValue", validationResults);
          }

        });

    return validationResults;
  }

  private void validateTriggerDWExpression(TriggerDescriptor triggerDescriptor,
                                           String expression, String[] implicitInputs, String expressionName,
                                           List<ValidationResult> results) {


    try {
      ExpressionHandlerUtils.compileDataWeaveScript(expression, implicitInputs);
    } catch (CompilationException e) {
      String shortErrorMessage =
          e.getMessage().contains("Location:") ? e.getMessage().substring(0, e.getMessage().indexOf("Location:")).trim()
              : e.getMessage();
      results.add(getValidationError(triggerDescriptor, expressionName, expression, shortErrorMessage));
    }

  }

  private ValidationResult getValidationError(TriggerDescriptor triggerDescriptor,
                                              String expressionName, String expression, String e) {
    String detail = format("The expression '%s' inside the trigger '%s', node '%s' is not valid. \nError: "
        + "%s", expression, triggerDescriptor.getName(),
                           expressionName, e);
    return new ValidationResult(this, detail, triggerDescriptor.getLocation());
  }

}

