/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.ConnectorModelBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.TriggerBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.BaseEndpointDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DescriptorElement;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TriggerDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorConnectorModelBuilderValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;

import java.util.List;
import java.util.Optional;
import java.util.stream.Stream;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.getApiOperation;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isEmpty;

public class TriggerPathAndMethodMustExistInApiOrOperationAdapterRule extends DescriptorConnectorModelBuilderValidationRule {

  public TriggerPathAndMethodMustExistInApiOrOperationAdapterRule() {
    super("Trigger PATH and METHOD must exist in the API specification or and OperationId in an operation adapter.", EMPTY,
          ERROR);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor,
                                         ConnectorModelBuilder connectorModelBuilder) {

    List<TriggerDescriptor> triggerDescriptors = connectorDescriptor.getTriggers();
    if (triggerDescriptors != null && !triggerDescriptors.isEmpty()) {
      return triggerDescriptors.stream()
          .flatMap(x -> validate(apiModel, x, connectorModelBuilder, connectorDescriptor))
          .collect(toList());
    }

    return emptyList();
  }

  private Stream<ValidationResult> validate(APIModel apiModel, TriggerDescriptor triggerDescriptor,
                                            ConnectorModelBuilder connectorModelBuilder,
                                            ConnectorDescriptor connectorDescriptor) {
    BaseEndpointDescriptor baseEndpoint = triggerDescriptor.getBaseEndpoint();
    if (baseEndpoint == null) {
      // can be a L3 description that has no baseEndpoint defined. Try to verify if a trigger was defined previously
      // and take its BaseEndpoint
      Optional<TriggerBuilder> parentTriggerBuilder = connectorModelBuilder.getTriggerBuilders()
          .stream()
          .filter(x -> x.getMethod() != null
              && !isEmpty(x.getPath())
              && x.getName().equals(triggerDescriptor.getName()))
          .findFirst();
      if (parentTriggerBuilder.isPresent()) {
        if (!apiModel.findOperation(parentTriggerBuilder.get().getPath(), parentTriggerBuilder.get().getMethod().name())
            .isPresent())
          return Stream.of(getValidationError(triggerDescriptor, triggerDescriptor));
        return Stream.empty();
      }
    }
    APIOperationModel apiOperation = getApiOperation(apiModel, connectorDescriptor, baseEndpoint);
    if (apiOperation == null)
      return Stream.of(getValidationError(triggerDescriptor, baseEndpoint == null ? triggerDescriptor : baseEndpoint));
    return Stream.empty();
  }

  private ValidationResult getValidationError(TriggerDescriptor triggerDescriptor, DescriptorElement elementForLocation) {
    StringBuilder detail = new StringBuilder("Trigger " + triggerDescriptor.getName());
    if ((triggerDescriptor.getBaseEndpoint() != null)) {
      detail.append(" with Operation baseEndpoint: '").append(triggerDescriptor.getBaseEndpoint());
    }
    detail.append("' does not exist in the API spec or any operation adapter's operation Id");

    return new ValidationResult(this, detail.toString(), elementForLocation.getLocation());
  }
}
